<?php
/**
 * Readability REST API Controller
 *
 * Handles readability analysis endpoints.
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Modules\Content\Readability;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * ReadabilityController class
 */
class ReadabilityController extends BaseController {
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->rest_base = 'readability-score';
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // GET /readability-score/{post_id}
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<post_id>\d+)',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_readability_score'],
                    'permission_callback' => [$this, 'check_permission_edit_posts'],
                    'args'                => [
                        'post_id' => [
                            'description' => __('Post ID to analyze', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                            'minimum'     => 1,
                        ],
                    ],
                ],
            ]
        );
    }
    
    /**
     * Get readability score for a post
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function get_readability_score(WP_REST_Request $request) {
        return $this->try_callback(function() use ($request) {
            $post_id = (int) $request->get_param('post_id');
            
            // Get post content
            $post_content = get_post_field('post_content', $post_id, 'raw');
            if (!is_string($post_content)) {
                $post_content = '';
            }
            
            // Check if post exists
            if (is_wp_error($post_content)) {
                return $this->get_error_response(
                    __('Post not found.', 'prorank-seo'),
                    404,
                    'post_not_found'
                );
            }
            
            // Handle empty content
            if (empty($post_content)) {
                return $this->get_success_response([
                    'score' => null,
                    'level' => null,
                    'grade' => null,
                    'message' => __('No content to analyze.', 'prorank-seo'),
                ]);
            }

            if (!class_exists(Readability::class)) {
                return $this->get_success_response([
                    'score' => null,
                    'level' => null,
                    'grade' => null,
                    'message' => __('Readability engine is unavailable.', 'prorank-seo'),
                ]);
            }
            
            // Strip HTML tags and clean content
            $cleaned_content = $this->clean_content($post_content);
            
            // Calculate readability score
            $score = Readability::calculate_flesch_kincaid($cleaned_content);
            
            // Prepare response data
            $response_data = [
                'score' => $score,
                'level' => null,
                'grade' => null,
                'stats' => null,
            ];
            
            if ($score !== null) {
                $response_data['level'] = Readability::get_readability_level($score);
                $response_data['grade'] = Readability::get_grade_level($score);

                // Count stats
                $word_count = str_word_count($cleaned_content);
                $sentence_count = preg_match_all('/[.!?]+/', $cleaned_content, $matches);
                $paragraph_count = preg_match_all('/\n\s*\n/', $post_content, $matches) + 1;

                // Add stats for UI display
                $response_data['stats'] = [
                    'word_count' => $word_count,
                    'sentence_count' => max(1, $sentence_count),
                    'paragraph_count' => max(1, $paragraph_count),
                    'estimated_reading_time' => $this->estimate_reading_time($cleaned_content),
                ];
            }
            
            return $this->get_success_response($response_data);
        });
    }
    
    /**
     * Clean content for analysis
     *
     * Removes HTML tags, shortcodes, and excessive whitespace.
     *
     * @param string $content Raw post content.
     * @return string Cleaned content.
     */
    private function clean_content(string $content): string {
        // Strip shortcodes
        $content = strip_shortcodes($content);
        
        // Strip HTML tags
        $content = wp_strip_all_tags($content);
        
        // Decode HTML entities
        $content = html_entity_decode($content, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // Remove multiple spaces and normalize whitespace
        $content = preg_replace('/\s+/', ' ', $content);
        
        // Trim
        $content = trim($content);
        
        return $content;
    }
    
    /**
     * Estimate reading time
     *
     * @param string $content Cleaned content.
     * @return int Estimated minutes to read.
     */
    private function estimate_reading_time(string $content): int {
        // Average reading speed: 200-250 words per minute
        // We'll use 225 as a middle ground
        $words_per_minute = 225;
        
        // Count words
        $word_count = str_word_count($content);
        
        // Calculate reading time in minutes
        $reading_time = ceil($word_count / $words_per_minute);
        
        return max(1, (int) $reading_time);
    }
}
