<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Performance Settings REST API Controller
 *
 * Handles performance module settings endpoints
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use ProRank\SEO\Modules\Performance\CriticalCssGenerator;

/**
 * Performance Settings controller class
 */
class PerformanceSettingsController extends BaseController {

    /**
     * Rest base
     *
     * @var string
     */
    protected $rest_base = 'performance';

    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Performance Beta toggle endpoint
        register_rest_route(
            $this->namespace,
            '/performance-beta',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_beta_status'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_beta_status'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'value' => [
                            'required' => true,
                            'type'     => 'boolean',
                        ],
                    ],
                ],
            ]
        );

        // RUM (Real User Monitoring) toggle
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/rum-settings',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_rum_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_rum_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'enabled' => [
                            'required' => true,
                            'type'     => 'boolean',
                        ],
                    ],
                ],
            ]
        );

        // Cache Settings (Page Cache, CDN, Preload, etc.)
        // Uses /performance/cache-settings to avoid SettingsController catch-all
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/cache-settings',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_cache_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_cache_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'settings' => [
                            'required' => true,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );

        // Image Optimization Settings
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/image-optimization/settings',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_image_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_image_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'settings' => [
                            'required' => true,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );

        // Asset Optimization Settings
        // Uses /performance/asset-settings to avoid SettingsController catch-all
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/asset-settings',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_asset_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_asset_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'settings' => [
                            'required' => true,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );

        // Database Optimization Settings
        register_rest_route(
            $this->namespace,
            '/settings/database',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_database_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_database_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'settings' => [
                            'required' => true,
                            'type'     => 'object',
                        ],
                    ],
                ],
            ]
        );

        // Image optimization stats
        register_rest_route(
            $this->namespace,
            '/performance/image-optimization/stats',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_image_stats'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // RUM ingest (INP/CLS/LCP)
        // This endpoint is intentionally public (__return_true) because RUM (Real User Monitoring)
        // collects anonymous performance metrics (Core Web Vitals) from site visitors.
        // No sensitive data is accepted or stored - only timing metrics for performance analysis.
        register_rest_route(
            $this->namespace,
            '/performance/rum',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'ingest_rum'],
                    'permission_callback' => [$this, 'check_rum_permission'],
                ],
            ]
        );

        // Image bulk optimization
        register_rest_route(
            $this->namespace,
            '/performance/image-optimization/bulk',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'bulk_optimize_images'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Critical CSS generation
        register_rest_route(
            $this->namespace,
            '/performance/critical-css/generate',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'generate_critical_css'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'target' => [
                            'required' => false,
                            'type'     => 'string',
                            'default'  => 'important',
                            'enum'     => ['important', 'all', 'custom'],
                        ],
                    ],
                ],
            ]
        );

        // Critical CSS queue (background)
        register_rest_route(
            $this->namespace,
            '/performance/critical-css/queue',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'queue_critical_css'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );


        // Critical CSS results
        register_rest_route(
            $this->namespace,
            '/performance/critical-css/results',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_critical_css_results'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Critical CSS rollback
        register_rest_route(
            $this->namespace,
            '/performance/critical-css/rollback',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'rollback_critical_css'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );


        // Font stats
        register_rest_route(
            $this->namespace,
            '/performance/font-stats',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_font_stats'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Database optimization actions
        register_rest_route(
            $this->namespace,
            '/database/optimize',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'optimize_database'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Database stats
        register_rest_route(
            $this->namespace,
            '/database/stats',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_database_stats'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Server info
        register_rest_route(
            $this->namespace,
            '/server/info',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_server_info'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // htaccess cache serving status
        register_rest_route(
            $this->namespace,
            '/performance/htaccess/status',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_htaccess_status'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Install htaccess rules
        register_rest_route(
            $this->namespace,
            '/performance/htaccess/install',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'install_htaccess_rules'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );


        // Nginx cache config
        register_rest_route(
            $this->namespace,
            '/performance/nginx-config',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_nginx_config'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        // Remove htaccess rules
        register_rest_route(
            $this->namespace,
            '/performance/htaccess/remove',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'remove_htaccess_rules'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
    }

    /**
     * Get performance beta status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_beta_status(WP_REST_Request $request): WP_REST_Response {
        $value = get_option('prorank_performance_beta', false);

        return new WP_REST_Response([
            'value' => (bool) $value,
        ], 200);
    }

    /**
     * Update performance beta status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function update_beta_status(WP_REST_Request $request): WP_REST_Response {
        $value = $request->get_param('value');
        update_option('prorank_performance_beta', $value);

        return new WP_REST_Response([
            'success' => true,
            'value' => $value,
        ], 200);
    }

    /**
     * Get RUM collection setting
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_rum_settings(WP_REST_Request $request): WP_REST_Response {
        $enabled = get_option('prorank_rum_enabled', false);

        return new WP_REST_Response([
            'enabled' => (bool) $enabled,
        ], 200);
    }

    /**
     * Update RUM collection setting
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function update_rum_settings(WP_REST_Request $request): WP_REST_Response {
        $enabled = (bool) $request->get_param('enabled');
        update_option('prorank_rum_enabled', $enabled, false);

        if (!$enabled) {
            update_option('prorank_rum_buffer', [], false);
        }

        return new WP_REST_Response([
            'success' => true,
            'enabled' => $enabled,
        ], 200);
    }

    /**
     * Get cache settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_cache_settings(WP_REST_Request $request): WP_REST_Response {
        $defaults = [
            // Page Cache
            'cache_enabled' => false,
            'cache_lifetime' => 3600,
            'cache_exclude_urls' => '',

            // CDN
            'cdn_provider' => 'none',
            'cloudflare_zone_id' => '',
            'cloudflare_api_token' => '',
            'bunny_pullzone_id' => '',
            'bunny_api_key' => '',

            // Preload (Pro)
            'preload_enabled' => false,
            'preload_mode' => 'sitemap',
            'preload_schedule' => 'daily',

            // Link Preloading
            'link_preload_enabled' => false,
            'link_preload_delay' => 50,

            // DNS Prefetch
            'dns_prefetch_enabled' => false,
            'dns_prefetch_hosts' => '',

            // Font Preloading
            'font_preload_enabled' => false,
            'font_preload_urls' => '',

            // Browser Cache
            'browser_cache_enabled' => false,
            'browser_cache_images' => '31536000',
            'browser_cache_css' => '31536000',
            'browser_cache_js' => '31536000',
            'browser_cache_fonts' => '31536000',
            'browser_cache_html' => '3600',

            // JS Optimization
            'js_defer' => false,
            'js_delay' => false,
            'js_exclude' => '',

            // CSS Optimization
            'css_minify' => false,
            'css_combine' => false,
            'css_async' => false,
            'css_exclude' => '',
        ];

        $settings = get_option('prorank_cache_settings', $defaults);

        return new WP_REST_Response([
            'settings' => wp_parse_args($settings, $defaults),
        ], 200);
    }

    /**
     * Update cache settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function update_cache_settings(WP_REST_Request $request): WP_REST_Response {
        $settings = $request->get_param('settings');

        if (!is_array($settings)) {
            return new WP_REST_Response([
                'error' => __('Invalid settings format', 'prorank-seo'),
            ], 400);
        }

        update_option('prorank_cache_settings', $settings);

        // Clear dashboard stats cache so changes reflect immediately
        delete_transient('prorank_dashboard_stats');

        return new WP_REST_Response([
            'success' => true,
            'settings' => $settings,
        ], 200);
    }

    /**
     * Get image optimization settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_image_settings(WP_REST_Request $request): WP_REST_Response {
        $defaults = [
            'webp_enabled' => true,
            'webp_quality' => 85,
            'avif_enabled' => false,
            'avif_quality' => 80,
            'jxl_enabled' => false,
            'jxl_quality' => 80,
            'jpegli_enabled' => false,
            'png_quality' => 80,
            'png_pipeline' => false,
            'smart_compression' => true,
            'compression_type' => 'glossy',
            'optimize_on_upload' => true,
            'background_mode' => false,
            'backup_originals' => true,
            'lazyload_images' => true,
            'lazyload_iframes' => false,
            'lazyload_videos' => false,
            'lazyload_threshold' => 200,
            'cdn_enabled' => false,
            'cdn_url' => '',
            'exclude_paths' => '',
            // Advanced settings
            'png_to_jpeg' => true,
            'cmyk_to_rgb' => true,
            'remove_exif' => true,
            'optimize_pdf' => false,
            'generate_retina' => false,
            'no_ai_training' => false,
            'delivery_method' => 'picture',
        ];

        $settings = get_option('prorank_image_optimization_settings', []);
        $settings = wp_parse_args(is_array($settings) ? $settings : [], $defaults);

        return new WP_REST_Response([
            'settings' => $settings,
        ], 200);
    }

    /**
     * Update image optimization settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function update_image_settings(WP_REST_Request $request): WP_REST_Response {
        $incoming = $request->get_param('settings');

        if (!is_array($incoming)) {
            return new WP_REST_Response([
                'error' => __('Invalid settings format', 'prorank-seo'),
            ], 400);
        }

        $defaults = [
            'webp_enabled' => true,
            'webp_quality' => 85,
            'avif_enabled' => false,
            'avif_quality' => 80,
            'jxl_enabled' => false,
            'jxl_quality' => 80,
            'jpegli_enabled' => false,
            'png_quality' => 80,
            'png_pipeline' => false,
            'smart_compression' => true,
            'compression_type' => 'glossy',
            'optimize_on_upload' => true,
            'background_mode' => false,
            'backup_originals' => true,
            'lazyload_images' => true,
            'lazyload_iframes' => false,
            'lazyload_videos' => false,
            'lazyload_threshold' => 200,
            'cdn_enabled' => false,
            'cdn_url' => '',
            'exclude_paths' => '',
            // Advanced settings
            'png_to_jpeg' => true,
            'cmyk_to_rgb' => true,
            'remove_exif' => true,
            'optimize_pdf' => false,
            'generate_retina' => false,
            'no_ai_training' => false,
            'delivery_method' => 'picture',
        ];

        // Merge with existing settings to avoid losing keys on partial updates
        $existing = get_option('prorank_image_optimization_settings', []);
        $settings = wp_parse_args(is_array($existing) ? $existing : [], $defaults);
        $settings = array_merge($settings, $incoming);

        // Validation
        $settings['compression_type'] = in_array($settings['compression_type'], ['lossless', 'glossy', 'lossy'], true)
            ? $settings['compression_type']
            : $defaults['compression_type'];

        $settings['delivery_method'] = in_array($settings['delivery_method'], ['picture', 'htaccess', 'cdn', 'none'], true)
            ? $settings['delivery_method']
            : $defaults['delivery_method'];

        foreach (['webp_quality', 'avif_quality', 'jxl_quality', 'png_quality'] as $key) {
            $settings[$key] = isset($settings[$key]) ? max(0, min(100, (int) $settings[$key])) : $defaults[$key];
        }

        $settings['lazyload_threshold'] = isset($settings['lazyload_threshold'])
            ? max(0, min(1000, (int) $settings['lazyload_threshold']))
            : $defaults['lazyload_threshold'];

        $settings['cdn_url'] = isset($settings['cdn_url']) ? esc_url_raw($settings['cdn_url']) : $defaults['cdn_url'];
        $settings['exclude_paths'] = isset($settings['exclude_paths']) ? sanitize_textarea_field($settings['exclude_paths']) : $defaults['exclude_paths'];

        // Cast remaining booleans
        foreach ([
            'webp_enabled',
            'avif_enabled',
            'jxl_enabled',
            'jpegli_enabled',
            'png_pipeline',
            'smart_compression',
            'optimize_on_upload',
            'background_mode',
            'backup_originals',
            'lazyload_images',
            'lazyload_iframes',
            'lazyload_videos',
            'cdn_enabled',
            'png_to_jpeg',
            'cmyk_to_rgb',
            'remove_exif',
            'optimize_pdf',
            'generate_retina',
            'no_ai_training',
        ] as $bool_key) {
            $settings[$bool_key] = !empty($settings[$bool_key]);
        }

        update_option('prorank_image_optimization_settings', $settings);

        return new WP_REST_Response([
            'success' => true,
            'settings' => $settings,
        ], 200);
    }

    /**
     * Get asset optimization settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_asset_settings(WP_REST_Request $request): WP_REST_Response {
        $defaults = [
            // Critical CSS
            'critical_css_enabled' => false,
            'critical_css_mode' => 'manual',
            'critical_css_manual' => '',
            'critical_css_target' => 'important',
            // JavaScript Minification
            'js_minify_enabled' => false,
            'js_minify_inline' => false,
            // Advanced Script Management
            'js_advanced_enabled' => false,
            'js_advanced_rules' => [],
            // Partytown / Third-Party Script Isolation
            'partytown_enabled' => false,
            'partytown_auto_detect' => true,
            'partytown_categories' => ['analytics', 'advertising', 'chat'],
            // INP Optimizer
            'inp_optimizer_enabled' => true,
            'inp_optimizer_monitoring' => true,
            'inp_optimizer_auto_optimize' => true,
            'inp_optimizer_sample_rate' => 10,
            // Font Optimization
            'host_google_fonts_locally' => false,
            'font_display_swap' => false,
            'font_subsetting_enabled' => false,
            // Video Optimization
            'video_optimize_youtube' => true,
            'video_optimize_vimeo' => true,
        ];

        $settings = get_option('prorank_asset_optimization_settings', $defaults);

        return new WP_REST_Response([
            'settings' => wp_parse_args($settings, $defaults),
        ], 200);
    }

    /**
     * Update asset optimization settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function update_asset_settings(WP_REST_Request $request): WP_REST_Response {
        $settings = $request->get_param('settings');

        if (!is_array($settings)) {
            return new WP_REST_Response([
                'error' => __('Invalid settings format', 'prorank-seo'),
            ], 400);
        }

        // Merge with existing to preserve unspecified keys
        $existing = get_option('prorank_asset_optimization_settings', []);
        $merged = wp_parse_args($settings, is_array($existing) ? $existing : []);

        // Cast booleans
        foreach ([
            'critical_css_enabled',
            'js_minify_enabled',
            'js_minify_inline',
            'js_advanced_enabled',
            'partytown_enabled',
            'partytown_auto_detect',
            'inp_optimizer_enabled',
            'inp_optimizer_monitoring',
            'inp_optimizer_auto_optimize',
            'host_google_fonts_locally',
            'font_display_swap',
            'font_subsetting_enabled',
            'video_optimize_youtube',
            'video_optimize_vimeo',
        ] as $bool_key) {
            if (isset($merged[$bool_key])) {
                $merged[$bool_key] = (bool) $merged[$bool_key];
            }
        }

        update_option('prorank_asset_optimization_settings', $merged);

        // Clear dashboard stats cache so changes reflect immediately
        delete_transient('prorank_dashboard_stats');

        return new WP_REST_Response([
            'success' => true,
            'settings' => $merged,
        ], 200);
    }

    /**
     * Get database optimization settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_database_settings(WP_REST_Request $request): WP_REST_Response {
        $defaults = [
            'auto_cleanup_enabled' => false,
            'cleanup_schedule' => 'weekly',
            'cleanup_items' => [
                'revisions' => true,
                'drafts' => false,
                'trash' => true,
                'spam_comments' => true,
                'unapproved_comments' => false,
                'transients' => true,
                'expired_transients' => true,
                'orphaned_meta' => false,
            ],
            'revision_limit' => 5,
            'trash_days' => 30,
            'opcache_enabled' => false,
            'memory_optimization' => false,
            'heartbeat_control' => false,
            'heartbeat_frequency' => 60,
            'admin_ajax_optimization' => false,
        ];

        $settings = get_option('prorank_database_settings', $defaults);

        return new WP_REST_Response([
            'settings' => wp_parse_args($settings, $defaults),
        ], 200);
    }

    /**
     * Update database optimization settings
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function update_database_settings(WP_REST_Request $request): WP_REST_Response {
        $settings = $request->get_param('settings');

        if (!is_array($settings)) {
            return new WP_REST_Response([
                'error' => __('Invalid settings format', 'prorank-seo'),
            ], 400);
        }

        update_option('prorank_database_settings', $settings);

        return new WP_REST_Response([
            'success' => true,
            'settings' => $settings,
        ], 200);
    }

    /**
     * Get image optimization statistics
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_image_stats(WP_REST_Request $request): WP_REST_Response {
        // Use module stats as single source of truth
        $stats = [
            'total_images' => 0,
            'optimized_images' => 0,
            'saved_bytes' => 0,
            'optimization_percentage' => 0,
            'backup_size' => 0,
            'backup_size_formatted' => '',
        ];

        $plugin = \ProRank\SEO\Plugin::get_instance();
        if ($plugin && $plugin->is_initialized()) {
            $module = $plugin->modules()->get_module('image_optimization');
            if ($module && method_exists($module, 'get_stats')) {
                $module_stats = $module->get_stats();
                $stats['total_images'] = (int) ($module_stats['total'] ?? 0);
                $stats['optimized_images'] = (int) ($module_stats['optimized'] ?? 0);
                $stats['saved_bytes'] = (int) ($module_stats['total_savings'] ?? $module_stats['bytes_saved'] ?? 0);
                $stats['optimization_percentage'] = $stats['total_images'] > 0
                    ? round(($stats['optimized_images'] / $stats['total_images']) * 100, 2)
                    : 0;
                if (!empty($module_stats['backup_size'])) {
                    $stats['backup_size'] = (int) $module_stats['backup_size'];
                    $stats['backup_size_formatted'] = $module_stats['backup_size_formatted'] ?? '';
                }
            }
        }

        return new WP_REST_Response([
            'stats' => $stats,
        ], 200);
    }

    /**
     * Ingest lightweight RUM metrics (INP/CLS/LCP)
     */
    public function check_rum_permission(): bool {
        if ( function_exists( 'prorank_is_rum_enabled' ) && ! prorank_is_rum_enabled() ) {
            return false;
        }

        return true;
    }

    public function ingest_rum(WP_REST_Request $request): WP_REST_Response {
        if ( function_exists( 'prorank_is_rum_enabled' ) && ! prorank_is_rum_enabled() ) {
            return new WP_REST_Response(
                [
                    'success' => false,
                    'message' => __( 'RUM collection is disabled.', 'prorank-seo' ),
                ],
                403
            );
        }
        $data = $request->get_json_params();
        $sample = [
            'url' => isset($data['url']) ? esc_url_raw($data['url']) : '',
            'lcp' => isset($data['lcp']) ? (int) $data['lcp'] : 0,
            'cls' => isset($data['cls']) ? (float) $data['cls'] : 0,
            'inp' => isset($data['inp']) ? (int) $data['inp'] : 0,
            'ts'  => time(),
        ];

        // Basic rate limit (per IP, per minute)
        $ip = \prorank_get_server_var( 'REMOTE_ADDR' );
        if (!empty($ip)) {
            $rl_key = 'prorank_rum_rl_' . md5($ip);
            $count = (int) get_transient($rl_key);
            if ($count >= 60) {
                return new WP_REST_Response(['success' => false, 'message' => 'Rate limited'], 429);
            }
            set_transient($rl_key, $count + 1, MINUTE_IN_SECONDS);
        }

        $sample['url'] = $this->normalize_site_url($sample['url']);

        if (empty($sample['url'])) {
            return new WP_REST_Response(['success' => false, 'message' => 'Invalid'], 400);
        }

        // Clamp values to reasonable bounds (helps avoid poisoning / huge payloads)
        $sample['lcp'] = max(0, min(60000, (int) $sample['lcp']));
        $sample['inp'] = max(0, min(60000, (int) $sample['inp']));
        $sample['cls'] = max(0.0, min(10.0, (float) $sample['cls']));

        $buffer = get_option('prorank_rum_buffer', []);
        if (!is_array($buffer)) {
            $buffer = [];
        }
        array_unshift($buffer, $sample);
        $buffer = array_slice($buffer, 0, 50); // keep last 50 samples
        update_option('prorank_rum_buffer', $buffer, false);

        return new WP_REST_Response(['success' => true], 200);
    }

    /**
     * Normalize and scope a URL to this site (prevents storing arbitrary external URLs)
     */
    private function normalize_site_url(string $url): string {
        $url = trim($url);
        if ($url === '') {
            return '';
        }

        $parsed = wp_parse_url($url);
        if ($parsed === false) {
            return '';
        }

        $site_host = wp_parse_url(home_url(), PHP_URL_HOST);
        $host = $parsed['host'] ?? '';

        if (!empty($host) && !empty($site_host) && strtolower($host) !== strtolower($site_host)) {
            return '';
        }

        $path = $parsed['path'] ?? '/';
        return $path !== '' ? $path : '/';
    }

    /**
     * Start bulk image optimization
     *
     * Bulk optimization runs locally on this server.
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function bulk_optimize_images(WP_REST_Request $request): WP_REST_Response {
        // Schedule bulk optimization
        if (!wp_next_scheduled('prorank_bulk_image_optimization')) {
            wp_schedule_single_event(time() + 10, 'prorank_bulk_image_optimization');
        }

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Bulk optimization started in background', 'prorank-seo'),
        ], 200);
    }

    /**
     * Generate critical CSS
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function generate_critical_css(WP_REST_Request $request): WP_REST_Response {
        $target = $request->get_param('target');

        try {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank] Starting Critical CSS generation, target: ' . $target);
            }
            $generator = new CriticalCssGenerator();
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank] CriticalCssGenerator instantiated');
            }
            $results = $generator->generate($target);
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank] Critical CSS generation complete');
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => __('Critical CSS generated successfully.', 'prorank-seo'),
                'results' => $results,
            ], 200);
        } catch (\Throwable $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('[ProRank] Critical CSS Error: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine());
            }
            return new WP_REST_Response([
                'success' => false,
                'message' => esc_html($e->getMessage()),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ], 500);
        }
    }

    /**
     * Queue critical CSS generation (background)
     */
    public function queue_critical_css(WP_REST_Request $request): WP_REST_Response {
        update_option(CriticalCssGenerator::STATUS_OPTION, 'queued');
        if (!wp_next_scheduled('prorank_generate_critical_css')) {
            wp_schedule_single_event(time() + 5, 'prorank_generate_critical_css', ['target' => 'important']);
        }
        return new WP_REST_Response([
            'success' => true,
            'status' => 'queued',
        ], 200);
    }

    /**
     * Get critical CSS results
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_critical_css_results(WP_REST_Request $request): WP_REST_Response {
        $results = CriticalCssGenerator::get_results();
        $status = CriticalCssGenerator::get_status();

        return new WP_REST_Response([
            'status' => $status,
            'results' => $results,
        ], 200);
    }

    /**
     * Roll back to previous critical CSS results
     */
    public function rollback_critical_css(WP_REST_Request $request): WP_REST_Response {
        $generator = new CriticalCssGenerator();
        $ok = $generator->rollback();

        return new WP_REST_Response([
            'success' => (bool) $ok,
        ], $ok ? 200 : 404);
    }

    /**
     * Optimize database tables
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function optimize_database(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;

        $tables_optimized = 0;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);

        foreach ($tables as $table) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("OPTIMIZE TABLE {$table[0]}");
            $tables_optimized++;
        }

        return new WP_REST_Response([
            'success' => true,
            'tables_optimized' => $tables_optimized,
            'message' => sprintf(
                /* translators: %s: numeric value */
                __('%d tables optimized', 'prorank-seo'), $tables_optimized),
        ], 200);
    }

    /**
     * Get database statistics
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_database_stats(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;

        // Get table count and size
        $tables = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT
                    COUNT(*) as count,
                    SUM(data_length + index_length) as size
                FROM information_schema.TABLES
                WHERE table_schema = %s",
                $wpdb->dbname
            )
        );

        // Get revision count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $revisions_count = $wpdb->get_var("
            SELECT COUNT(*) FROM {$wpdb->posts}
            WHERE post_type = 'revision'
        ");

        // Get trash count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $trash_count = $wpdb->get_var("
            SELECT COUNT(*) FROM {$wpdb->posts}
            WHERE post_status = 'trash'
        ");

        // Get transient count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $transients_count = $wpdb->get_var("
            SELECT COUNT(*) FROM {$wpdb->options}
            WHERE option_name LIKE '_transient_%'
        ");

        return new WP_REST_Response([
            'stats' => [
                'table_count' => (int) ($tables[0]->count ?? 0),
                'total_size' => $this->format_bytes((int) ($tables[0]->size ?? 0)),
                'revisions_count' => (int) $revisions_count,
                'trash_count' => (int) $trash_count,
                'transients_count' => (int) $transients_count,
            ],
        ], 200);
    }

    /**
     * Format bytes to human readable
     *
     * @param int $bytes Size in bytes
     * @return string Formatted size
     */
    /**
     * Get server information
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_server_info(WP_REST_Request $request): WP_REST_Response {
        global $wpdb;

        // PHP info
        $php_version = phpversion();
        $memory_limit = ini_get('memory_limit');
        $max_execution_time = ini_get('max_execution_time');
        $upload_max_filesize = ini_get('upload_max_filesize');
        $post_max_size = ini_get('post_max_size');

        // MySQL/MariaDB info
        $mysql_version = $wpdb->get_var('SELECT VERSION()');
        $mysql_type = (stripos($mysql_version, 'mariadb') !== false) ? 'MariaDB' : 'MySQL';

        // OPcache status
        $opcache_enabled = function_exists('opcache_get_status') && @opcache_get_status() !== false;
        $opcache_info = [];
        if ($opcache_enabled) {
            $status = @opcache_get_status();
            if ($status) {
                $opcache_info = [
                    'enabled' => true,
                    'memory_usage' => isset($status['memory_usage']['used_memory']) 
                        ? round($status['memory_usage']['used_memory'] / 1048576, 2) . ' MB' 
                        : 'N/A',
                    'hit_rate' => isset($status['opcache_statistics']['opcache_hit_rate'])
                        ? round($status['opcache_statistics']['opcache_hit_rate'], 2) . '%'
                        : 'N/A',
                ];
            }
        }

        // Object cache status
        $object_cache_type = 'none';
        $object_cache_info = [];
        if (wp_using_ext_object_cache()) {
            global $wp_object_cache;
            if (class_exists('Redis') && isset($wp_object_cache) && method_exists($wp_object_cache, 'redis_instance')) {
                $object_cache_type = 'redis';
                try {
                    $redis = $wp_object_cache->redis_instance();
                    if ($redis && method_exists($redis, 'info')) {
                        $info = $redis->info();
                        $object_cache_info = [
                            'version' => $info['redis_version'] ?? 'N/A',
                            'memory' => isset($info['used_memory_human']) ? $info['used_memory_human'] : 'N/A',
                        ];
                    }
                } catch (\Exception $e) {
                    // Redis connection failed
                }
            } elseif (class_exists('Memcached')) {
                $object_cache_type = 'memcached';
            } elseif (class_exists('Memcache')) {
                $object_cache_type = 'memcache';
            } else {
                $object_cache_type = 'external';
            }
        }

        // Web server info
        $server_software = \prorank_get_server_var( 'SERVER_SOFTWARE' );
        if ($server_software === '') {
            $server_software = 'Unknown';
        }

        return new WP_REST_Response([
            'info' => [
                'php_version' => $php_version,
                'memory_limit' => $memory_limit,
                'max_execution_time' => $max_execution_time . 's',
                'upload_max_filesize' => $upload_max_filesize,
                'post_max_size' => $post_max_size,
                'mysql_version' => $mysql_version,
                'mysql_type' => $mysql_type,
                'server_software' => $server_software,
                'opcache_enabled' => $opcache_enabled,
                'opcache_info' => $opcache_info,
                'object_cache_type' => $object_cache_type,
                'object_cache_info' => $object_cache_info,
            ],
        ], 200);
    }

    /**
     * Format bytes to human readable
     *
     * @param int $bytes Size in bytes
     * @return string Formatted size
     */
    private function format_bytes(int $bytes): string {
        if ($bytes < 1024) {
            return $bytes . ' B';
        } elseif ($bytes < 1048576) {
            return round($bytes / 1024, 2) . ' KB';
        } elseif ($bytes < 1073741824) {
            return round($bytes / 1048576, 2) . ' MB';
        } else {
            return round($bytes / 1073741824, 2) . ' GB';
        }
    }

    /**
     * Apply unused CSS optimizations by generating purged CSS files
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function apply_unused_css_optimizations(WP_REST_Request $request): WP_REST_Response {
        try {
            // Get scan results
            $results = get_option('prorank_unused_css_results', null);

            if (!$results || empty($results['stylesheets'])) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('No scan results found. Please run a CSS scan first.', 'prorank-seo'),
                ], 400);
            }

            $used_selectors = $results['used_selectors'] ?? [];
            if (!is_array($used_selectors) || empty($used_selectors)) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('Scan results are missing used selectors. Please re-run the scan.', 'prorank-seo'),
                ], 400);
            }

            // Create cache directory (persist across page-cache clears)
            $upload_dir = wp_upload_dir();
            $cache_dir = trailingslashit($upload_dir['basedir']) . 'prorank-cache/css';
            if (!is_dir($cache_dir)) {
                wp_mkdir_p($cache_dir);
            }
            global $wp_filesystem;
            if ( ! function_exists( 'WP_Filesystem' ) ) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            WP_Filesystem();

            if (!is_dir($cache_dir) || !$wp_filesystem || !$wp_filesystem->is_writable($cache_dir)) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('CSS cache directory is not writable. Please check file permissions for wp-content/uploads/prorank-cache/css.', 'prorank-seo'),
                ], 500);
            }

            $files_generated = 0;
            $total_saved = 0;
            $safelist = $this->get_safelist_patterns();
            if (!empty($results['safelist_applied']) && is_array($results['safelist_applied'])) {
                $safelist = array_merge($safelist, $results['safelist_applied']);
            }

            $stored_safelist = get_option('prorank_unused_css_safelist', '');
            if (is_string($stored_safelist) && $stored_safelist !== '') {
                $lines = array_filter(array_map('trim', preg_split('/[\r\n]+/', $stored_safelist)));
                if (!empty($lines)) {
                    $safelist = array_merge($safelist, $lines);
                }
            }
            $site_host = wp_parse_url(home_url('/'), PHP_URL_HOST);

            $used_selectors_by_context = $results['used_selectors_by_context'] ?? [];
            $has_context_sets = is_array($used_selectors_by_context) && !empty($used_selectors_by_context);

            $context_files_generated = 0;
            $context_bytes_saved = 0;

            // Cache fetched stylesheets to avoid repeated HTTP requests.
            $original_css_cache = [];

            $stylesheets_by_context = $results['stylesheets_by_context'] ?? [];
            $has_stylesheets_by_context = is_array($stylesheets_by_context) && !empty($stylesheets_by_context);

            // Build the list of stylesheet URLs to optimize.
            $stylesheets_srcs = [];
            if ($has_stylesheets_by_context) {
                foreach ($stylesheets_by_context as $context => $sheet_srcs) {
                    if (!is_array($sheet_srcs)) {
                        continue;
                    }
                    foreach ($sheet_srcs as $src) {
                        if (is_string($src) && $src !== '' && strpos($src, 'inline-') !== 0) {
                            $stylesheets_srcs[$src] = true;
                        }
                    }
                }
            } else {
                foreach ($results['stylesheets'] as $stylesheet) {
                    $src = $stylesheet['src'] ?? '';
                    if (is_string($src) && $src !== '' && strpos($src, 'inline-') !== 0) {
                        $stylesheets_srcs[$src] = true;
                    }
                }
            }

            // 1) Generate global UCSS files (apply everywhere): md5($src).css
            foreach (array_keys($stylesheets_srcs) as $src) {

                // Convert relative URLs to absolute for requests + hashing consistency.
                if (strpos($src, '//') === 0) {
                    $src = 'https:' . $src;
                } elseif (strpos($src, '/') === 0) {
                    $src = home_url($src);
                }

                // Skip Google Fonts (handled separately)
                if (strpos($src, 'fonts.googleapis.com') !== false ||
                    strpos($src, 'fonts.gstatic.com') !== false) {
                    continue;
                }

                // Skip external stylesheets (different host).
                $src_host = wp_parse_url($src, PHP_URL_HOST);
                if (!empty($src_host) && !empty($site_host) && $src_host !== $site_host) {
                    continue;
                }

                if (!array_key_exists($src, $original_css_cache)) {
                    $response = wp_remote_get($src, [
                        'timeout' => 30,
                        'user-agent' => 'ProRank CSS Optimizer/1.0',
                    ]);

                    if (is_wp_error($response)) {
                        $original_css_cache[$src] = null;
                    } else {
                        $code = (int) wp_remote_retrieve_response_code($response);
                        $content_type = (string) wp_remote_retrieve_header($response, 'content-type');
                        $body = wp_remote_retrieve_body($response);

                        // Skip non-CSS responses (e.g. 404 HTML) to avoid generating broken cache files.
                        if ($code !== 200) {
                            $original_css_cache[$src] = null;
                        } elseif ($content_type !== '' && stripos($content_type, 'text/css') === false) {
                            $original_css_cache[$src] = null;
                        } else {
                            $original_css_cache[$src] = $body;
                        }
                    }
                }

                $original_css = $original_css_cache[$src];
                if (empty($original_css)) {
                    continue;
                }

                // Save to cache
                $original_size = strlen($original_css);

                // 1) Global UCSS file (applies everywhere): ucss-{md5($src)}.css
                $optimized_css = $this->purge_unused_css($original_css, $used_selectors, $safelist);
                $optimized_size = strlen($optimized_css);
                $saved = ($original_size - $optimized_size);

                if ($saved > 0) {
                    $cache_key = md5($src);
                    $cache_file = $cache_dir . '/ucss-' . $cache_key . '.css';
                    $written = file_put_contents($cache_file, $optimized_css);
                    if ($written !== false) {
                        $gzipped = gzencode($optimized_css, 9);
                        if ($gzipped !== false) {
                            @file_put_contents($cache_file . '.gz', $gzipped);
                        }
                        $files_generated++;
                        $total_saved += $saved;
                    }
                }
            }

            // 2) Generate context-specific UCSS files for ProRank combined CSS:
            // ucss-{md5($src . '|' . $context)}.css (frontend prefers these when present)
            if ($has_context_sets) {
                foreach ($used_selectors_by_context as $context => $context_selectors) {
                    if (!is_string($context) || $context === '' || !is_array($context_selectors) || empty($context_selectors)) {
                        continue;
                    }

                    // Prefer live HTML extraction for the current context so we don't optimize stale combined files.
                    $context_url = $this->get_sample_url_for_context($context);
                    $context_sheets = [];

                    if (is_string($context_url) && $context_url !== '') {
                        $html_response = wp_remote_get($context_url, [
                            'timeout' => 30,
                            'user-agent' => 'ProRank CSS Optimizer/1.0',
                        ]);

                        if (!is_wp_error($html_response) && (int) wp_remote_retrieve_response_code($html_response) === 200) {
                            $html = wp_remote_retrieve_body($html_response);
                            if (is_string($html) && $html !== '') {
                                $context_sheets = $this->extract_stylesheet_hrefs_from_html($html);
                            }
                        }
                    }

                    // Fallback to scan result mapping if HTML fetch failed.
                    if (empty($context_sheets) && $has_stylesheets_by_context) {
                        $context_sheets = $stylesheets_by_context[$context] ?? [];
                    }

                    if (!is_array($context_sheets) || empty($context_sheets)) {
                        continue;
                    }

                    foreach ($context_sheets as $src) {
                        if (!is_string($src) || $src === '' || !$this->is_prorank_combined_stylesheet($src)) {
                            continue;
                        }

                        // Normalize URL to match global processing and cache keys.
                        if (strpos($src, '//') === 0) {
                            $src = 'https:' . $src;
                        } elseif (strpos($src, '/') === 0) {
                            $src = home_url($src);
                        }

                        // Skip Google Fonts (handled separately)
                        if (strpos($src, 'fonts.googleapis.com') !== false ||
                            strpos($src, 'fonts.gstatic.com') !== false) {
                            continue;
                        }

                        // Skip external stylesheets (different host).
                        $src_host = wp_parse_url($src, PHP_URL_HOST);
                        if (!empty($src_host) && !empty($site_host) && $src_host !== $site_host) {
                            continue;
                        }

                        if (!array_key_exists($src, $original_css_cache)) {
                            $response = wp_remote_get($src, [
                                'timeout' => 30,
                                'user-agent' => 'ProRank CSS Optimizer/1.0',
                            ]);

                            if (is_wp_error($response)) {
                                $original_css_cache[$src] = null;
                            } else {
                                $code = (int) wp_remote_retrieve_response_code($response);
                                $content_type = (string) wp_remote_retrieve_header($response, 'content-type');
                                $body = wp_remote_retrieve_body($response);

                                if ($code !== 200) {
                                    $original_css_cache[$src] = null;
                                } elseif ($content_type !== '' && stripos($content_type, 'text/css') === false) {
                                    $original_css_cache[$src] = null;
                                } else {
                                    $original_css_cache[$src] = $body;
                                }
                            }
                        }

                        $original_css = $original_css_cache[$src];
                        if (empty($original_css)) {
                            continue;
                        }

                        $original_size = strlen($original_css);
                        $context_css = $this->purge_unused_css($original_css, $context_selectors, $safelist);
                        $context_size = strlen($context_css);
                        $context_saved = ($original_size - $context_size);

                        if ($context_saved <= 0) {
                            continue;
                        }

                        $context_key = md5($src . '|' . $context);
                        $context_file = $cache_dir . '/ucss-' . $context_key . '.css';
                        $written = file_put_contents($context_file, $context_css);
                        if ($written === false) {
                            continue;
                        }

                        $gzipped = gzencode($context_css, 9);
                        if ($gzipped !== false) {
                            @file_put_contents($context_file . '.gz', $gzipped);
                        }

                        $context_files_generated++;
                        $context_bytes_saved += $context_saved;
                    }
                }
            }

            // Mark optimizations as applied
            update_option('prorank_unused_css_applied', [
                'applied_at' => current_time('mysql'),
                'files_generated' => $files_generated,
                'bytes_saved' => $total_saved,
                'context_files_generated' => $context_files_generated,
                'context_bytes_saved' => $context_bytes_saved,
            ]);

            // Enable the optimization in settings
            $settings = get_option('prorank_asset_optimization_settings', []);
            $settings['unused_css_enabled'] = true;
            update_option('prorank_asset_optimization_settings', $settings);

            // ProRank page cache serves static HTML; purge it so new HTML can include swapped CSS URLs.
            $purged_files = $this->purge_prorank_html_cache_for_current_host();

            return new WP_REST_Response([
                'success' => true,
                'message' => sprintf(
                    __('Generated %1\$d optimized CSS files (+%2\$d context files). Saving %3\$s.', 'prorank-seo'),
                    $files_generated,
                    $context_files_generated,
                    $this->format_bytes($total_saved + $context_bytes_saved)
                ),
                'files_generated' => $files_generated,
                'bytes_saved' => $total_saved,
                'context_files_generated' => $context_files_generated,
                'context_bytes_saved' => $context_bytes_saved,
                'html_cache_purged_files' => $purged_files,
            ], 200);

        } catch (\Throwable $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => esc_html($e->getMessage()),
            ], 500);
        }
    }

    /**
     * Purge ProRank HTML page cache for the current site host.
     *
     * Only clears the HTML page cache. CSS assets are stored under `wp-content/uploads/prorank-cache/`.
     *
     * @return int Number of files deleted
     */
    private function purge_prorank_html_cache_for_current_host(): int {
        $host = wp_parse_url(home_url('/'), PHP_URL_HOST);
        if (empty($host)) {
            return 0;
        }

        $cache_dir = prorank_page_cache_dir($host);
        if (!is_dir($cache_dir)) {
            return 0;
        }

        $deleted = 0;
        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($cache_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::CHILD_FIRST
        );

        foreach ($iterator as $file) {
            if ($file->isDir()) {
                if ( $wp_filesystem ) {
                    $wp_filesystem->rmdir($file->getPathname());
                }
                continue;
            }

            if (wp_delete_file($file->getPathname())) {
                $deleted++;
            }
        }

        if ( $wp_filesystem ) {
            $wp_filesystem->rmdir($cache_dir);
        }
        return $deleted;
    }

    /**
     * Check if a stylesheet is ProRank's combined CSS output.
     *
     * @param string $src Stylesheet URL
     * @return bool
     */
    private function is_prorank_combined_stylesheet(string $src): bool {
        return strpos($src, '/prorank-cache/css/combined-') !== false
            || strpos($src, '/wp-content/uploads/prorank-cache/css/combined-') !== false;
    }

    /**
     * Get a representative URL for a scan context.
     *
     * Used to refresh context stylesheets (especially ProRank combined CSS) at apply-time so
     * we don't generate optimizations for stale cache filenames.
     *
     * @param string $context Context identifier (e.g. front_page, single_post)
     * @return string|null URL or null when not available
     */
    private function get_sample_url_for_context(string $context): ?string {
        switch ($context) {
            case 'front_page':
                return home_url('/');

            case 'single_page':
                $front_id = (int) get_option('page_on_front');
                $pages = get_posts([
                    'post_type' => 'page',
                    'post_status' => 'publish',
                    'posts_per_page' => 1,
                    'post__not_in' => $front_id ? [$front_id] : [],
                    'orderby' => 'date',
                    'order' => 'DESC',
                ]);
                if (!empty($pages)) {
                    $url = get_permalink($pages[0]);
                    return is_string($url) && $url !== '' ? $url : home_url('/');
                }
                return home_url('/');

            case 'single_post':
                $posts = get_posts([
                    'post_type' => 'post',
                    'post_status' => 'publish',
                    'posts_per_page' => 1,
                    'orderby' => 'date',
                    'order' => 'DESC',
                ]);
                if (!empty($posts)) {
                    $url = get_permalink($posts[0]);
                    return is_string($url) && $url !== '' ? $url : null;
                }
                return null;

            case 'archive_category':
                $cats = get_categories([
                    'hide_empty' => false,
                    'number' => 1,
                ]);
                if (!empty($cats)) {
                    $url = get_category_link($cats[0]);
                    return is_string($url) && $url !== '' ? $url : null;
                }
                return null;

            case 'archive':
                // Try the posts page when "Your homepage displays" is set to a static page.
                $page_for_posts = (int) get_option('page_for_posts');
                if ($page_for_posts) {
                    $url = get_permalink($page_for_posts);
                    return is_string($url) && $url !== '' ? $url : home_url('/');
                }
                return home_url('/');

            default:
                // Best-effort fallback.
                return home_url('/');
        }
    }

    /**
     * Extract stylesheet hrefs from HTML.
     *
     * @param string $html Raw HTML
     * @return array<int,string> Unique hrefs
     */
    private function extract_stylesheet_hrefs_from_html(string $html): array {
        $hrefs = [];

        if (preg_match_all("#<link[^>]+rel=[\"']stylesheet[\"'][^>]*href=[\"']([^\"']+)[\"']#i", $html, $matches)) {
            foreach (($matches[1] ?? []) as $href) {
                if (!is_string($href) || $href === '') {
                    continue;
                }
                $href = html_entity_decode($href, ENT_QUOTES);
                $hrefs[$href] = true;
            }
        }

        return array_keys($hrefs);
    }

    /**
     * Clear unused CSS optimizations
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function clear_unused_css_optimizations(WP_REST_Request $request): WP_REST_Response {
        try {
            // Clear cache directory
            $upload_dir = wp_upload_dir();
            $cache_dir = trailingslashit($upload_dir['basedir']) . 'prorank-cache/css';
            if (is_dir($cache_dir)) {
                $files = array_merge(
                    glob($cache_dir . '/ucss-*.css') ?: [],
                    glob($cache_dir . '/ucss-*.css.gz') ?: []
                );
                foreach ($files as $file) {
                    wp_delete_file($file);
                }
            }

            // Clear applied status
            delete_option('prorank_unused_css_applied');
            delete_option('prorank_unused_css_results');

            // Disable in settings
            $settings = get_option('prorank_asset_optimization_settings', []);
            $settings['unused_css_enabled'] = false;
            update_option('prorank_asset_optimization_settings', $settings);

            // Clear page cache
            if (function_exists('rocket_clean_domain')) {
                rocket_clean_domain();
            }
            wp_cache_flush();
            $purged_files = $this->purge_prorank_html_cache_for_current_host();

            return new WP_REST_Response([
                'success' => true,
                'message' => __('Optimizations cleared. Original CSS files restored.', 'prorank-seo'),
                'html_cache_purged_files' => $purged_files,
            ], 200);

        } catch (\Throwable $e) {
            return new WP_REST_Response([
                'success' => false,
                'message' => esc_html($e->getMessage()),
            ], 500);
        }
    }

    /**
     * Purge unused CSS from stylesheet
     *
     * @param string $css Original CSS
     * @param array $used_selectors Used selectors
     * @param array $safelist Safelist patterns
     * @return string Purged CSS
     */
    private function purge_unused_css(string $css, array $used_selectors, array $safelist): string {
        $used = $this->build_used_token_sets($used_selectors);
        $safelist = $this->normalize_safelist_patterns($safelist);

        // Remove comments
        $css = preg_replace('/\/\*.*?\*\//s', '', $css);

        $purged = '';

        // Keep @charset/@import statements (order-sensitive).
        $css = preg_replace_callback('/@(charset|import)[^;]*;/i', function($match) use (&$purged) {
            $purged .= $match[0] . "\n";
            return '';
        }, $css);

        // Keep @font-face blocks.
        $css = preg_replace_callback('/@font-face\\s*\\{[^}]*\\}/i', function($match) use (&$purged) {
            $purged .= $match[0] . "\n";
            return '';
        }, $css);

        // Keep @keyframes blocks (incl. vendor prefixes).
        $css = preg_replace_callback('/@(?:-webkit-|-moz-|-o-)?keyframes\\s+[^\\{]+\\{(?:[^{}]+|\\{[^{}]*\\})*\\}/i', function($match) use (&$purged) {
            $purged .= $match[0] . "\n";
            return '';
        }, $css);

        // Process @media rules
        $css = preg_replace_callback('/@media[^{]*\{((?:[^{}]+|\{[^{}]*\})*)\}/s', function($match) use ($used, $safelist, &$purged) {
            $media_query = $match[0];
            $inner_css = $match[1];

            // Purge inner rules
            $inner_purged = $this->purge_rules($inner_css, $used, $safelist);

            if (!empty(trim($inner_purged))) {
                $purged .= str_replace($match[1], $inner_purged, $media_query) . "\n";
            }
            return '';
        }, $css);

        // Process @supports/@container/@layer similarly to @media.
        $css = preg_replace_callback('/@(supports|container|layer)[^{]*\{((?:[^{}]+|\\{[^{}]*\\})*)\\}/s', function($match) use ($used, $safelist, &$purged) {
            $at_rule = $match[0];
            $inner_css = $match[2] ?? '';

            $inner_purged = $this->purge_rules($inner_css, $used, $safelist);
            if (!empty(trim($inner_purged))) {
                $purged .= str_replace($inner_css, $inner_purged, $at_rule) . "\n";
            }
            return '';
        }, $css);

        // Process remaining rules
        $purged .= $this->purge_rules($css, $used, $safelist);

        // Minify
        $purged = preg_replace('/\s+/', ' ', $purged);
        $purged = preg_replace('/\s*([{}:;,])\s*/', '$1', $purged);
        $purged = str_replace(';}', '}', $purged);

        return trim($purged);
    }

    /**
     * Purge individual CSS rules
     *
     * @param string $css CSS rules
     * @param array $used_selectors Used selectors
     * @param array $safelist Safelist patterns
     * @return string Purged rules
     */
    private function purge_rules(string $css, array $used_selectors, array $safelist): string {
        $purged = '';

        preg_match_all('/([^{]+)\{([^}]+)\}/s', $css, $matches, PREG_SET_ORDER);

        foreach ($matches as $match) {
            $selector_group = trim($match[1]);
            $declarations = trim($match[2]);

            if (empty($selector_group) || empty($declarations)) {
                continue;
            }

            // Split selector groups and keep only the matching selectors.
            $selectors = array_filter(array_map('trim', explode(',', $selector_group)));
            if (empty($selectors)) {
                continue;
            }

            $kept = [];
            foreach ($selectors as $selector) {
                if ($this->should_keep_single_selector($selector, $used_selectors, $safelist)) {
                    $kept[] = $selector;
                }
            }

            if (!empty($kept)) {
                $purged .= implode(',', $kept) . '{' . $declarations . '}';
            }
        }

        return $purged;
    }

    /**
     * Build lookup sets for extracted selector tokens.
     *
     * @param array $used_selectors Used selectors from scan (classes/ids/tags/attrs)
     * @return array{classes: array<string,true>, ids: array<string,true>, tags: array<string,true>, attrs: array<string,true>}
     */
    private function build_used_token_sets(array $used_selectors): array {
        $used = [
            'classes' => [],
            'ids' => [],
            'tags' => [],
            'attrs' => [],
        ];

        foreach ($used_selectors as $selector) {
            if (!is_string($selector)) {
                continue;
            }

            $selector = trim($selector);
            if ($selector === '') {
                continue;
            }

            $first = $selector[0] ?? '';
            if ($first === '.') {
                $name = substr($selector, 1);
                if ($name !== '') {
                    $used['classes'][$name] = true;
                }
                continue;
            }

            if ($first === '#') {
                $name = substr($selector, 1);
                if ($name !== '') {
                    $used['ids'][$name] = true;
                }
                continue;
            }

            if ($first === '[' && substr($selector, -1) === ']') {
                $name = trim(substr($selector, 1, -1));
                if ($name !== '') {
                    $used['attrs'][$name] = true;
                }
                continue;
            }

            // Treat the remainder as a tag token (from HTML extraction).
            $tag = strtolower($selector);
            if (preg_match('/^[a-z][a-z0-9-]*$/', $tag)) {
                $used['tags'][$tag] = true;
            }
        }

        return $used;
    }

    /**
     * Normalize safelist patterns into token patterns (class/id/attribute names).
     *
     * @param array $patterns Patterns from UI/settings
     * @return array Normalized patterns
     */
    private function normalize_safelist_patterns(array $patterns): array {
        $normalized = [];

        foreach ($patterns as $pattern) {
            if (!is_string($pattern)) {
                continue;
            }

            $pattern = trim($pattern);
            if ($pattern === '') {
                continue;
            }

            // Convert selector-like patterns into token patterns.
            if ($pattern[0] === '.' || $pattern[0] === '#') {
                $pattern = substr($pattern, 1);
            } elseif ($pattern[0] === '[' && substr($pattern, -1) === ']') {
                $pattern = trim(substr($pattern, 1, -1));
            }

            $pattern = trim($pattern);
            if ($pattern === '') {
                continue;
            }

            $normalized[$pattern] = true;
        }

        return array_keys($normalized);
    }

    /**
     * Check whether a token matches the safelist.
     *
     * @param string $token Token to test (class/id/attr/tag name)
     * @param array $safelist Safelist token patterns (fnmatch)
     * @return bool
     */
    private function token_is_safelisted(string $token, array $safelist): bool {
        foreach ($safelist as $pattern) {
            if (!is_string($pattern) || $pattern === '') {
                continue;
            }
            if (fnmatch($pattern, $token)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check if a single (non-comma) selector should be kept.
     *
     * Uses a PurgeCSS-style heuristic: keep selector if all its tokens exist in
     * the extracted used tokens, with support for basic :is()/:where() groups.
     *
     * @param string $selector Single selector (no commas)
     * @param array $used_sets Used token sets from build_used_token_sets()
     * @param array $safelist Safelist token patterns
     * @return bool
     */
    private function should_keep_single_selector(string $selector, array $used_sets, array $safelist): bool {
        $selector = trim($selector);
        if ($selector === '') {
            return false;
        }

        // Always keep global/root selectors and common vendor pseudo selectors.
        if ($selector === ':root' || $selector === 'html' || $selector === 'body' || $selector === '*') {
            return true;
        }
        if (strpos($selector, '::-webkit-') !== false || strpos($selector, '::-moz-') !== false) {
            return true;
        }

        // Evaluate :is()/:where() groups (OR lists). Every group must have a match.
        if (preg_match_all('/:((?:is|where))\(([^()]*)\)/i', $selector, $groups, PREG_SET_ORDER)) {
            foreach ($groups as $group) {
                $content = $group[2] ?? '';
                $alternatives = array_filter(array_map('trim', explode(',', $content)));
                if (empty($alternatives)) {
                    continue;
                }

                $any = false;
                foreach ($alternatives as $alt) {
                    if ($this->should_keep_single_selector($alt, $used_sets, $safelist)) {
                        $any = true;
                        break;
                    }
                }

                if (!$any) {
                    return false;
                }
            }
        }

        // Remove :not(...) blocks (their contents are NOT required to exist).
        $normalized = preg_replace('/:not\([^)]*\)/i', '', $selector);

        // Remove :is(...) and :where(...) blocks from required token extraction (handled above).
        $normalized = preg_replace('/:((?:is|where))\([^)]*\)/i', '', $normalized);

        // Extract tokens.
        $classes = [];
        $ids = [];
        $attrs = [];
        $tags = [];

        if (preg_match_all('/\.([_a-zA-Z][\\w-]*)/', $normalized, $m)) {
            $classes = $m[1];
        }
        if (preg_match_all('/#([_a-zA-Z][\\w-]*)/', $normalized, $m)) {
            $ids = $m[1];
        }
        if (preg_match_all('/\\[([^\\s~|\\^\\$\\*=\\]]+)/', $normalized, $m)) {
            $attrs = $m[1];
        }

        $has_simple = !empty($classes) || !empty($ids) || !empty($attrs);
        if (!$has_simple) {
            // Only consider tag tokens when selector has no class/id/attr (e.g. "a", "h1").
            if (preg_match_all('/(?:^|[\\s>+~])([a-zA-Z][\\w-]*)/', $normalized, $m)) {
                $tags = array_map('strtolower', $m[1]);
            }
        }

        // Require all class tokens.
        foreach ($classes as $class) {
            if (isset($used_sets['classes'][$class])) {
                continue;
            }
            if ($this->token_is_safelisted($class, $safelist)) {
                continue;
            }
            return false;
        }

        // Require all ID tokens.
        foreach ($ids as $id) {
            if (isset($used_sets['ids'][$id])) {
                continue;
            }
            if ($this->token_is_safelisted($id, $safelist)) {
                continue;
            }
            return false;
        }

        // Require all attribute tokens.
        foreach ($attrs as $attr) {
            if (isset($used_sets['attrs'][$attr])) {
                continue;
            }
            if ($this->token_is_safelisted($attr, $safelist)) {
                continue;
            }
            return false;
        }

        // Require tag tokens only when we have no class/id/attr tokens. If we have no tag
        // tokens (e.g. empty selector after normalization), keep only if it’s a known safe case.
        if (!$has_simple) {
            if (empty($tags)) {
                return false;
            }

            // If the scan did not include tag extraction, don’t drop
            // tag-only selectors.
            if (empty($used_sets['tags'])) {
                return true;
            }

            foreach ($tags as $tag) {
                if ($tag === '') {
                    continue;
                }
                if (isset($used_sets['tags'][$tag])) {
                    continue;
                }
                if ($this->token_is_safelisted($tag, $safelist)) {
                    continue;
                }
                return false;
            }
        }

        return true;
    }

    /**
     * Get safelist token patterns for CSS purging
     *
     * @return array
     */
    private function get_safelist_patterns(): array {
        return [
            // WordPress + block editor
            'wp-*',
            'wp-block-*',
            'has-*',
            'is-*',

            // Common JS/dynamic states
            'js-*',
            'no-js',
            'active',
            'open',
            'closed',
            'show',
            'hide',
            'hidden',
            'visible',
            'loading',
            'loaded',
            'disabled',
            'selected',
            'checked',
            'expanded',
            'collapsed',

            // Popular plugins/themes
            'woocommerce*',
            'wc-*',
            'elementor*',
            'e-*',
            'ast-*',
            'astra-*',

            // Attribute patterns
            'data-*',
            'aria-*',
            'role',
            'type',
        ];
    }

    /**
     * Get font optimization stats
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_font_stats(WP_REST_Request $request): WP_REST_Response {
        $upload_dir = wp_upload_dir();
        $fonts_dir = $upload_dir['basedir'] . '/prorank-fonts';

        $stats = [
            'total_fonts' => 0,
            'total_size' => 0,
            'google_fonts_found' => 0,
        ];

        // Count local font files
        if (is_dir($fonts_dir)) {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($fonts_dir, \RecursiveDirectoryIterator::SKIP_DOTS)
            );

            foreach ($iterator as $file) {
                if ($file->isFile() && $file->getFilename() !== 'index.php') {
                    $stats['total_fonts']++;
                    $stats['total_size'] += $file->getSize();
                }
            }
        }

        // Check for Google Fonts in registered styles
        global $wp_styles;
        if (!empty($wp_styles->registered)) {
            foreach ($wp_styles->registered as $style) {
                if (!empty($style->src) && is_string($style->src) && strpos($style->src, 'fonts.googleapis.com') !== false) {
                    $stats['google_fonts_found']++;
                }
            }
        }

        return new WP_REST_Response([
            'stats' => $stats,
        ], 200);
    }

    /**
     * Get htaccess cache serving status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_htaccess_status(WP_REST_Request $request): WP_REST_Response {
        $plugin = \ProRank\SEO\Plugin::get_instance();
        $module = $plugin->modules()->get_module('modern_cache');

        if (!$module) {
            // Fallback: return basic htaccess info without the module
            $htaccess_path = prorank_get_home_path() . '.htaccess';
            $exists = file_exists($htaccess_path);

            global $wp_filesystem;
            if ( ! function_exists( 'WP_Filesystem' ) ) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }
            WP_Filesystem();

            $home_path = prorank_get_home_path();
            $writable = $wp_filesystem && ($exists ? $wp_filesystem->is_writable($htaccess_path) : $wp_filesystem->is_writable($home_path));

            return new WP_REST_Response([
                'success' => true,
                'status' => [
                    'exists' => $exists,
                    'writable' => $writable,
                    'has_prorank_rules' => false,
                    'server' => \prorank_get_server_var( 'SERVER_SOFTWARE' ) ?: 'unknown',
                ],
            ], 200);
        }

        $status = $module->get_htaccess_status();

        return new WP_REST_Response([
            'success' => true,
            'status' => $status,
        ], 200);
    }

    /**
     * Install htaccess cache serving rules
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function install_htaccess_rules(WP_REST_Request $request): WP_REST_Response {
        $plugin = \ProRank\SEO\Plugin::get_instance();
        $module = $plugin->modules()->get_module('modern_cache');

        if (!$module) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Cache module not found', 'prorank-seo'),
            ], 404);
        }

        // Enable htaccess serving in settings
        $settings = get_option('prorank_cache_settings', []);
        $settings['htaccess_serving'] = true;
        update_option('prorank_cache_settings', $settings);

        $result = $module->install_htaccess_rules();

        return new WP_REST_Response([
            'success' => $result,
            'message' => $result
                ? __('htaccess rules installed successfully', 'prorank-seo')
                : __('Failed to install htaccess rules', 'prorank-seo'),
            'status' => $module->get_htaccess_status(),
        ], $result ? 200 : 500);
    }

    /**
     * Remove htaccess cache serving rules
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function remove_htaccess_rules(WP_REST_Request $request): WP_REST_Response {
        $plugin = \ProRank\SEO\Plugin::get_instance();
        $module = $plugin->modules()->get_module('modern_cache');

        if (!$module) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Cache module not found', 'prorank-seo'),
            ], 404);
        }

        // Disable htaccess serving in settings
        $settings = get_option('prorank_cache_settings', []);
        $settings['htaccess_serving'] = false;
        update_option('prorank_cache_settings', $settings);

        $result = $module->remove_htaccess_rules();

        return new WP_REST_Response([
            'success' => $result,
            'message' => $result
                ? __('htaccess rules removed successfully', 'prorank-seo')
                : __('Failed to remove htaccess rules', 'prorank-seo'),
            'status' => $module->get_htaccess_status(),
        ], $result ? 200 : 500);
    }

    /**
     * Get nginx cache config snippet
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_nginx_config(WP_REST_Request $request): WP_REST_Response {
        $cache_path = rtrim(prorank_page_cache_dir(), '/');
        $cache_url_path = rtrim(wp_make_link_relative(prorank_page_cache_url()), '/');
        
        $config = "# ProRank SEO - Nginx Cache Configuration\n";
        $config .= "# Add this to your nginx server block\n\n";
        $config .= "location / {\n";
        $config .= "    # Try cached files first\n";
        $config .= "    set \$cache_file '" . $cache_path . "\$request_uri/index.html';\n";
        $config .= "    if (-f \$cache_file) {\n";
        $config .= "        rewrite ^(.*)\$ " . $cache_url_path . "\$1/index.html break;\n";
        $config .= "    }\n";
        $config .= "    try_files \$uri \$uri/ /index.php?\$args;\n";
        $config .= "}\n";

        return new WP_REST_Response([
            'success' => true,
            'config' => $config,
            'cache_path' => $cache_path,
        ], 200);
    }
}
