<?php
/**
 * Modules REST API Controller
 *
 * Handles module management endpoints
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use ProRank\SEO\Core\ModuleManager;

/**
 * Modules controller class
 */
class ModulesController extends BaseController {
    
    /**
     * Rest base
     *
     * @var string
     */
    protected $rest_base = 'modules';
    
    /**
     * Module manager instance
     *
     * @var ModuleManager|null
     */
    private ?ModuleManager $module_manager = null;
    
    /**
     * Constructor
     */
    public function __construct() {
        // Module manager will be set when routes are registered
    }
    
    /**
     * Set module manager
     *
     * @param ModuleManager $module_manager Module manager instance
     * @return void
     */
    public function set_module_manager(ModuleManager $module_manager): void {
        $this->module_manager = $module_manager;
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Get all modules
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_modules'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Get module groups
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/groups',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_module_groups'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Toggle module group status
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/groups/(?P<group>[a-zA-Z0-9_-]+)/status',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'toggle_module_group'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'group' => [
                            'required'          => true,
                            'sanitize_callback' => 'sanitize_key',
                        ],
                        'enabled' => [
                            'required'          => true,
                            'type'              => 'boolean',
                        ],
                    ],
                ],
            ]
        );
        
        // Toggle module
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<module>[a-zA-Z0-9_-]+)/toggle',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'toggle_module'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'module' => [
                            'required'          => true,
                            'sanitize_callback' => 'sanitize_key',
                        ],
                        'enabled' => [
                            'required'          => true,
                            'type'              => 'boolean',
                        ],
                    ],
                ],
            ]
        );
        
        // Get module status
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<module>[a-zA-Z0-9_-]+)/status',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_module_status'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'module' => [
                            'required'          => true,
                            'sanitize_callback' => 'sanitize_key',
                        ],
                    ],
                ],
            ]
        );
        
        // Dashboard module status (simplified)
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/status',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_modules_status'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );

        // Get modules by parent group
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/by-group/(?P<group>[a-zA-Z0-9_-]+)',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_modules_by_group'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'group' => [
                            'required'          => true,
                            'sanitize_callback' => 'sanitize_key',
                        ],
                    ],
                ],
            ]
        );
    }
    
    /**
     * Get all modules
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_modules(WP_REST_Request $request) {
        if (!$this->module_manager) {
            return $this->get_error_response(
                __('Module manager not available', 'prorank-seo'),
                500
            );
        }
        
        try {
            $modules = $this->module_manager->get_modules();
            $module_data = [];

            foreach ($modules as $slug => $module) {
                if (!is_object($module)) {
                    continue;
                }

                $parent_slug = method_exists($module, 'get_parent_slug') ? $module->get_parent_slug() : null;
                $category = $parent_slug ?: __('General', 'prorank-seo');

                $module_data[] = [
                    'slug' => (string) $slug,
                    'name' => $module->get_name(),
                    'description' => $module->get_description(),
                    'tier' => 'free',
                    'parent_slug' => $parent_slug,
                    'category' => $category,
                    'icon' => 'dashicons-admin-generic',
                    'enabled' => $this->module_manager->is_module_enabled($slug),
                    'available' => true,
                    'locked' => false,
                    'features' => [],
                    'recommended' => false,
                    'core' => false,
                ];
            }

            return rest_ensure_response([
                'modules' => $module_data,
                'total' => count($module_data),
            ]);

        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get modules: %s', 'prorank-seo'),
                    esc_html($e->getMessage())
                ),
                500
            );
        }
    }
    
    /**
     * Toggle module activation
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function toggle_module(WP_REST_Request $request) {
        $module_slug = $request->get_param('module');
        $enabled = $request->get_param('enabled');
        
        if (!$this->module_manager) {
            return $this->get_error_response(
                __('Module manager not available', 'prorank-seo'),
                500
            );
        }
        
        try {
            // Check if module exists
            $modules = $this->module_manager->get_modules();
            if (!isset($modules[$module_slug])) {
                return $this->get_error_response(
                    sprintf(
                        /* translators: %s: module name */
                        __('Module "%s" not found', 'prorank-seo'),
                        $module_slug
                    ),
                    404
                );
            }
            
            // Toggle module
            if ($enabled) {
                $result = $this->module_manager->activate_module($module_slug);
            } else {
                $result = $this->module_manager->deactivate_module($module_slug);
            }
            
            if (!$result) {
                return $this->get_error_response(
                    sprintf(
                        /* translators: %s: module action (activate/deactivate) */
                        __('Failed to %s module', 'prorank-seo'),
                        $enabled ? 'activate' : 'deactivate'
                    ),
                    500
                );
            }
            
            // Trigger action
            do_action('prorank_seo_module_toggled', $module_slug, $enabled);
            
            return rest_ensure_response([
                'success' => true,
                'message' => sprintf(
                    /* translators: 1: module slug, 2: action (activated/deactivated) */
                    __('Module "%1$s" %2$s successfully', 'prorank-seo'),
                    $module_slug,
                    $enabled ? 'activated' : 'deactivated'
                ),
                'module' => $module_slug,
                'active' => $enabled,
            ]);
            
        } catch (\Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank SEO - Module toggle error: ' . $e->getMessage());
            }
            if (defined('WP_DEBUG') && WP_DEBUG) {
                prorank_log('ProRank SEO - Module toggle trace: ' . $e->getTraceAsString());
            }
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to toggle module: %s', 'prorank-seo'),
                    esc_html($e->getMessage())
                ),
                500
            );
        }
    }
    
    /**
     * Get module status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_module_status(WP_REST_Request $request) {
        $module_slug = $request->get_param('module');
        
        if (!$this->module_manager) {
            return $this->get_error_response(
                __('Module manager not available', 'prorank-seo'),
                500
            );
        }
        
        try {
            // Check if module exists
            $modules = $this->module_manager->get_modules();
            if (!isset($modules[$module_slug])) {
                return $this->get_error_response(
                    sprintf(
                        /* translators: %s: module name */
                        __('Module "%s" not found', 'prorank-seo'),
                        $module_slug
                    ),
                    404
                );
            }
            
            $module = $modules[$module_slug];
            $config = method_exists($module, 'get_config') ? $module->get_config() : [];
            
            return rest_ensure_response([
                'slug' => $module_slug,
                'active' => $this->module_manager->is_active($module_slug),
                'name' => $config['name'] ?? $module_slug,
                'tier' => $config['tier'] ?? 'free',
                'parent' => $config['parent'] ?? null,
                'parent_active' => $config['parent'] ? $this->module_manager->is_active($config['parent']) : true,
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get module status: %s', 'prorank-seo'),
                    esc_html($e->getMessage())
                ),
                500
            );
        }
    }
    
    /**
     * Get module manifest data
     *
     * In production we ship a static catalog (generated at build/release time)
     * so premium modules can still be shown as locked placeholders when the
     * Premium plugin is not installed/active.
     *
     * @return array<string, array<string, mixed>>
     */
    private function get_module_manifest_data(): array {
        $manifest_file = PRORANK_PLUGIN_DIR . 'config/module-catalog.json';

        if (!file_exists($manifest_file)) {
            return [];
        }

        $raw = prorank_read_file($manifest_file);
        if ($raw === '') {
            return [];
        }

        $decoded = json_decode($raw, true);
        if (!is_array($decoded) || !isset($decoded['modules']) || !is_array($decoded['modules'])) {
            return [];
        }

        return $decoded['modules'];
    }

    /**
     * Get module groups
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_module_groups(WP_REST_Request $request) {
        if (!$this->module_manager) {
            return $this->get_error_response(
                __('Module manager not available', 'prorank-seo'),
                500
            );
        }
        
        try {
            // Define main module groups
            $groups = [
                [
                    'slug' => 'on-page-seo',
                    'name' => __('On-Page SEO', 'prorank-seo'),
                    'description' => __('Content optimization, readability analysis, and internal linking', 'prorank-seo'),
                    'tier' => 'free',
                    'enabled' => $this->is_module_group_enabled('on-page-seo'),
                ],
                [
                    'slug' => 'technical-seo',
                    'name' => __('Technical SEO', 'prorank-seo'),
                    'description' => __('Sitemaps, redirects, robots.txt, and URL cleanup', 'prorank-seo'),
                    'tier' => 'free',
                    'enabled' => $this->is_module_group_enabled('technical-seo'),
                ],
                [
                    'slug' => 'performance',
                    'name' => __('Performance', 'prorank-seo'),
                    'description' => __('Speed optimization, caching, and Core Web Vitals', 'prorank-seo'),
                    'tier' => 'free',
                    'enabled' => $this->is_module_group_enabled('performance'),
                ],
            ];

            foreach ($groups as &$group) {
                $group['locked'] = false;
                $group['available'] = true;
            }
            unset($group);

            return rest_ensure_response([
                'groups' => $groups,
                'total' => count($groups),
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get module groups: %s', 'prorank-seo'),
                    esc_html($e->getMessage())
                ),
                500
            );
        }
    }
    
    /**
     * Toggle module group status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function toggle_module_group(WP_REST_Request $request) {
        $group_slug = $request->get_param('group');
        $enabled = $request->get_param('enabled');
        
        if (!$this->module_manager) {
            return $this->get_error_response(
                __('Module manager not available', 'prorank-seo'),
                500
            );
        }
        
        try {
            // Get all modules in this group
            $modules = $this->module_manager->get_modules();
            $affected_modules = [];

            foreach ($modules as $slug => $module) {
                if (!is_object($module) || !method_exists($module, 'get_parent_slug')) {
                    continue;
                }
                if ($module->get_parent_slug() === $group_slug) {
                    if ($enabled) {
                        $this->module_manager->activate_module($slug);
                    } else {
                        $this->module_manager->deactivate_module($slug);
                    }
                    $affected_modules[] = $slug;
                }
            }
            
            // Save group state
            $group_states = get_option('prorank_seo_module_groups', []);
            $group_states[$group_slug] = $enabled;
            update_option('prorank_seo_module_groups', $group_states);
            
            return rest_ensure_response([
                'success' => true,
                'message' => sprintf(
                    /* translators: 1: module group slug, 2: status (enabled/disabled) */
                    __('Module group "%1\$s" %2\$s successfully', 'prorank-seo'),
                    $group_slug,
                    $enabled ? 'enabled' : 'disabled'
                ),
                'group' => $group_slug,
                'enabled' => $enabled,
                'affected_modules' => $affected_modules,
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to toggle module group: %s', 'prorank-seo'),
                    esc_html($e->getMessage())
                ),
                500
            );
        }
    }
    
    /**
     * Check if module group is enabled
     *
     * @param string $group_slug Group slug
     * @return bool
     */
    private function is_module_group_enabled(string $group_slug): bool {
        $group_states = get_option('prorank_seo_module_groups', []);
        
        // Default to enabled for free tier groups
        if (!isset($group_states[$group_slug])) {
            return in_array($group_slug, ['on-page-seo', 'technical-seo', 'performance'], true);
        }
        
        return (bool) $group_states[$group_slug];
    }
    
    /**
     * Get modules by parent group
     *
     * Returns modules for the requested group. Uses the static catalog so
     * premium modules remain visible as locked placeholders when not loaded.
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_modules_by_group(WP_REST_Request $request) {
        $group_slug = $request->get_param('group');

        if (!$this->module_manager) {
            return $this->get_error_response(
                __('Module manager not available', 'prorank-seo'),
                500
            );
        }

        try {
            $loaded_modules = $this->module_manager->get_modules();

            $group_modules = [];

            foreach ($loaded_modules as $slug => $module) {
                if (!is_object($module)) {
                    continue;
                }
                $parent_slug = $module->get_parent_slug();
                if ($parent_slug !== $group_slug) {
                    continue;
                }

                $group_modules[] = [
                    'slug' => (string) $slug,
                    'name' => (string) $module->get_name(),
                    'description' => (string) $module->get_description(),
                    'tier' => 'free',
                    'parent_slug' => $parent_slug,
                    'icon' => 'dashicons-admin-generic',
                    'enabled' => $this->module_manager->is_module_enabled($slug),
                    'features' => [],
                ];
            }

            foreach ($group_modules as &$module) {
                $module['tier'] = 'free';
                $module['locked'] = false;
                $module['available'] = true;
            }
            unset($module);

            // Get group info
            $group_info = null;
            $groups_response = $this->get_module_groups(new WP_REST_Request());
            $groups = is_object($groups_response) && isset($groups_response->data['groups']) ? $groups_response->data['groups'] : [];
            foreach ($groups as $group) {
                if (($group['slug'] ?? null) === $group_slug) {
                    $group_info = $group;
                    break;
                }
            }

            return rest_ensure_response([
                'group' => $group_info,
                'modules' => $group_modules,
                'total' => count($group_modules),
            ]);

        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get modules by group: %s', 'prorank-seo'),
                    esc_html($e->getMessage())
                ),
                500
            );
        }
    }

    /**
     * Get module groups status for dashboard overview
     *
     * Returns high-level module groups with counts of enabled child modules
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_modules_status(WP_REST_Request $request) {
        if (!$this->module_manager) {
            return $this->get_error_response(
                __('Module manager not available', 'prorank-seo'),
                500
            );
        }

        try {
            $modules = $this->module_manager->get_modules();

            // Define main module groups
            $groups = [
                'on-page-seo' => [
                    'slug' => 'on-page-seo',
                    'name' => __('On-Page SEO', 'prorank-seo'),
                    'description' => __('Meta tags, schema, internal linking', 'prorank-seo'),
                    'total' => 0,
                    'enabled_count' => 0,
                ],
                'technical-seo' => [
                    'slug' => 'technical-seo',
                    'name' => __('Technical SEO', 'prorank-seo'),
                    'description' => __('Sitemaps, redirects, robots.txt', 'prorank-seo'),
                    'total' => 0,
                    'enabled_count' => 0,
                ],
                'performance' => [
                    'slug' => 'performance',
                    'name' => __('Performance', 'prorank-seo'),
                    'description' => __('Caching, optimization, Core Web Vitals', 'prorank-seo'),
                    'total' => 0,
                    'enabled_count' => 0,
                ],
            ];

            // Count modules per group
            foreach ($modules as $slug => $module) {
                try {
                    if (!is_object($module) || !method_exists($module, 'get_parent_slug')) {
                        continue;
                    }

                    $parent_slug = $module->get_parent_slug();

                    // Skip parent modules themselves
                    if (empty($parent_slug)) {
                        continue;
                    }

                    // Map parent slugs to groups
                    $group_key = $parent_slug;

                    if (isset($groups[$group_key])) {
                        $groups[$group_key]['total']++;
                        if ($this->module_manager->is_module_enabled($slug)) {
                            $groups[$group_key]['enabled_count']++;
                        }
                    }
                } catch (\Exception $e) {
                    continue;
                }
            }

            // Convert to array and calculate enabled status
            $module_groups = [];
            foreach ($groups as $group) {
                // Only include groups that have modules
                if ($group['total'] > 0) {
                    $group['enabled'] = $group['enabled_count'] > 0;
                    $module_groups[] = $group;
                }
            }

            // Sort by enabled count descending
            usort($module_groups, function ($a, $b) {
                return $b['enabled_count'] - $a['enabled_count'];
            });

            return rest_ensure_response([
                'modules' => $module_groups,
            ]);
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get module status: %s', 'prorank-seo'),
                    esc_html($e->getMessage())
                ),
                500
            );
        }
    }

    /**
     * Convert icon token to dashicon class
     *
     * @param string $icon_token Icon token from manifest
     * @return string Dashicon class
     */
    private function get_icon_class(string $icon_token): string {
        // Map icon tokens to dashicon classes
        $icon_map = [
            '--pr-icon-dashboard' => 'dashicons-dashboard',
            '--pr-icon-on-page' => 'dashicons-edit',
            '--pr-icon-titles-meta' => 'dashicons-tag',
            '--pr-icon-content-analysis' => 'dashicons-analytics',
            '--pr-icon-schema' => 'dashicons-editor-code',
            '--pr-icon-schema-advanced' => 'dashicons-editor-expand',
            '--pr-icon-technical' => 'dashicons-admin-tools',
            '--pr-icon-sitemap' => 'dashicons-networking',
            '--pr-icon-author-sitemap' => 'dashicons-admin-users',
            '--pr-icon-video-sitemap' => 'dashicons-video-alt3',
            '--pr-icon-redirects' => 'dashicons-randomize',
            '--pr-icon-redirect-log' => 'dashicons-text-page',
            '--pr-icon-performance' => 'dashicons-performance',
            '--pr-icon-cache' => 'dashicons-database',
            '--pr-icon-critical-css' => 'dashicons-art',
            '--pr-icon-analytics' => 'dashicons-chart-area',
            '--pr-icon-search-console' => 'dashicons-google',
            '--pr-icon-local' => 'dashicons-location',
            '--pr-icon-ai' => 'dashicons-lightbulb',
            '--pr-icon-ai-content' => 'dashicons-welcome-write-blog',
            '--pr-icon-ai-verify' => 'dashicons-yes-alt',
            '--pr-icon-woocommerce' => 'dashicons-cart',
            '--pr-icon-product-schema' => 'dashicons-products',
            '--pr-icon-social' => 'dashicons-share',
            '--pr-icon-tools' => 'dashicons-admin-tools',
            '--pr-icon-support' => 'dashicons-sos',
            '--pr-icon-white-label' => 'dashicons-admin-appearance',
            '--pr-icon-podcast' => 'dashicons-microphone',
            '--pr-icon-podcast-advanced' => 'dashicons-playlist-audio',
        ];
        
        return $icon_map[$icon_token] ?? 'dashicons-admin-generic';
    }
}
