<?php
/**
 * Meta REST API Controller
 *
 * Handles meta tags (title, description, canonical) via REST API.
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * MetaController class
 */
class MetaController extends BaseController {
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->rest_base = 'meta';
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // GET/POST /meta/{post_id}
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<post_id>\d+)',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_item'],
                    'permission_callback' => [$this, 'check_permission_edit_posts'],
                    'args'                => [
                        'post_id' => [
                            'description' => __('Post ID', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                            'minimum'     => 1,
                        ],
                    ],
                ],
                [
                    'methods'             => \WP_REST_Server::EDITABLE,
                    'callback'            => [$this, 'update_item'],
                    'permission_callback' => [$this, 'check_permission_edit_posts'],
                    'args'                => array_merge(
                        [
                            'post_id' => [
                                'description' => __('Post ID', 'prorank-seo'),
                                'type'        => 'integer',
                                'required'    => true,
                                'minimum'     => 1,
                            ],
                        ],
                        $this->get_meta_schema()
                    ),
                ],
            ]
        );

        // GET /meta/{post_id}/analysis-content
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/(?P<post_id>\d+)/analysis-content',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_analysis_content'],
                    'permission_callback' => [$this, 'check_permission_edit_posts'],
                    'args'                => [
                        'post_id' => [
                            'description' => __('Post ID', 'prorank-seo'),
                            'type'        => 'integer',
                            'required'    => true,
                            'minimum'     => 1,
                        ],
                    ],
                ],
            ]
        );
    }
    
    /**
     * Get meta data for a post
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function get_item($request) {
        return $this->try_callback(function() use ($request) {
            $post_id = (int) $request->get_param('post_id');
            
            // Verify post exists
            $post = get_post($post_id);
            if (!$post) {
                return $this->get_error_response(
                    __('Post not found.', 'prorank-seo'),
                    404,
                    'post_not_found'
                );
            }
            if (!current_user_can('edit_post', $post_id)) {
                return $this->get_error_response(
                    __('Insufficient permissions for this post.', 'prorank-seo'),
                    403,
                    'forbidden'
                );
            }
            
            // Get meta values with defaults
            $seo_title = get_post_meta($post_id, '_prorank_seo_title', true);
            $seo_description = get_post_meta($post_id, '_prorank_seo_description', true);
            $canonical_url = get_post_meta($post_id, '_prorank_seo_canonical_url', true);
            $meta_robots = get_post_meta($post_id, '_prorank_meta_robots', true);
            
            // Get social meta values
            $og_title = get_post_meta($post_id, '_prorank_og_title', true);
            $og_description = get_post_meta($post_id, '_prorank_og_description', true);
            $og_image_id = get_post_meta($post_id, '_prorank_og_image_id', true);
            $twitter_title = get_post_meta($post_id, '_prorank_twitter_title', true);
            $twitter_description = get_post_meta($post_id, '_prorank_twitter_description', true);
            $twitter_image_id = get_post_meta($post_id, '_prorank_twitter_image_id', true);
            $twitter_card_type = get_post_meta($post_id, '_prorank_twitter_card_type', true);
            
            // Get schema data
            $schema_data = get_post_meta($post_id, '_prorank_schema_data', true);

            // Get focus keyword
            $focus_keyword = get_post_meta($post_id, '_prorank_focus_keyword', true);
            
            // Provide defaults
            if (empty($seo_title)) {
                $seo_title = '';
            }
            if (empty($seo_description)) {
                $seo_description = '';
            }
            if (empty($canonical_url)) {
                $canonical_url = '';
            }
            if (!is_array($meta_robots)) {
                $meta_robots = [];
            }
            
            // Social defaults
            if (empty($og_title)) {
                $og_title = '';
            }
            if (empty($og_description)) {
                $og_description = '';
            }
            if (empty($og_image_id)) {
                $og_image_id = 0;
            }
            if (empty($twitter_title)) {
                $twitter_title = '';
            }
            if (empty($twitter_description)) {
                $twitter_description = '';
            }
            if (empty($twitter_image_id)) {
                $twitter_image_id = 0;
            }
            if (empty($twitter_card_type)) {
                $twitter_card_type = 'summary';
            }
            if (!is_array($schema_data)) {
                $schema_data = [];
            }
            if (empty($focus_keyword)) {
                $focus_keyword = '';
            }
            
            // Get image URLs
            $og_image_url = '';
            if ($og_image_id) {
                $og_image_url = wp_get_attachment_url((int) $og_image_id);
            }
            
            $twitter_image_url = '';
            if ($twitter_image_id) {
                $twitter_image_url = wp_get_attachment_url((int) $twitter_image_id);
            }
            
            // Get featured image as fallback
            $featured_image_url = '';
            $featured_image_id = get_post_thumbnail_id($post_id);
            if ($featured_image_id) {
                $featured_image_url = wp_get_attachment_url($featured_image_id);
            }
            
            return $this->get_success_response([
                'seo_title' => $seo_title,
                'seo_description' => $seo_description,
                'canonical_url' => $canonical_url,
                'meta_robots' => $meta_robots,
                // Focus keyword
                'focus_keyword' => $focus_keyword,
                // Social fields
                'og_title' => $og_title,
                'og_description' => $og_description,
                'og_image_id' => (int) $og_image_id,
                'og_image_url' => $og_image_url ?: '',
                'twitter_title' => $twitter_title,
                'twitter_description' => $twitter_description,
                'twitter_image_id' => (int) $twitter_image_id,
                'twitter_image_url' => $twitter_image_url ?: '',
                'twitter_card_type' => $twitter_card_type,
                // Schema data
                'schema_data' => $schema_data,
                // Internal linking data
                'is_pillar_content' => (bool) get_post_meta($post_id, '_prorank_is_pillar_content', true),
                'link_stats' => $this->get_post_link_stats($post_id, $post->post_content),
                'defaults' => [
                    'title' => $post->post_title,
                    'url' => get_permalink($post_id),
                ],
                'featured_image_url' => $featured_image_url ?: '',
            ]);
        });
    }

    /**
     * Get analysis content for a post (including Elementor content when available).
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function get_analysis_content($request) {
        return $this->try_callback(function() use ($request) {
            $post_id = (int) $request->get_param('post_id');

            $post = get_post($post_id);
            if (!$post) {
                return $this->get_error_response(
                    __('Post not found.', 'prorank-seo'),
                    404,
                    'post_not_found'
                );
            }
            if (!current_user_can('edit_post', $post_id)) {
                return $this->get_error_response(
                    __('Insufficient permissions for this post.', 'prorank-seo'),
                    403,
                    'forbidden'
                );
            }

            $content = (string) $post->post_content;
            $source = 'post_content';

            if ($this->is_content_empty($content)) {
                $elementor_rendered = $this->get_elementor_rendered_content($post_id);
                if (!$this->is_content_empty($elementor_rendered)) {
                    $content = $elementor_rendered;
                    $source = 'elementor_rendered';
                } else {
                    $elementor_text = $this->get_elementor_data_text($post_id);
                    if (!$this->is_content_empty($elementor_text)) {
                        $content = $elementor_text;
                        $source = 'elementor_data';
                    }
                }
            }

            $content = wp_kses_post($content);

            return $this->get_success_response([
                'content' => $content,
                'title'   => get_the_title($post_id),
                'source'  => $source,
            ]);
        });
    }

    /**
     * Determine if content is empty after stripping markup.
     */
    private function is_content_empty(string $content): bool {
        return trim(wp_strip_all_tags($content)) === '';
    }

    /**
     * Fetch Elementor rendered content if Elementor is active.
     */
    private function get_elementor_rendered_content(int $post_id): string {
        if (!class_exists('\\Elementor\\Plugin')) {
            return '';
        }

        try {
            $plugin = \Elementor\Plugin::$instance;
            if (!$plugin || !isset($plugin->frontend)) {
                return '';
            }
            return (string) $plugin->frontend->get_builder_content_for_display($post_id, true);
        } catch (\Throwable $e) {
            return '';
        }
    }

    /**
     * Extract text from Elementor data meta.
     */
    private function get_elementor_data_text(int $post_id): string {
        $raw = get_post_meta($post_id, '_elementor_data', true);
        if (empty($raw)) {
            return '';
        }

        $data = is_array($raw) ? $raw : json_decode((string) $raw, true);
        if (!is_array($data)) {
            return '';
        }

        $chunks = [];
        $this->collect_elementor_text($data, $chunks);
        if (empty($chunks)) {
            return '';
        }

        $text = trim(preg_replace('/\s+/', ' ', implode(' ', $chunks)));
        return $text;
    }

    /**
     * Recursively collect text values from Elementor element settings.
     */
    private function collect_elementor_text($node, array &$chunks): void {
        if (is_string($node)) {
            $clean = trim(wp_strip_all_tags($node));
            if ($clean !== '') {
                $chunks[] = $clean;
            }
            return;
        }

        if (!is_array($node)) {
            return;
        }

        if (isset($node['settings']) && is_array($node['settings'])) {
            foreach ($node['settings'] as $key => $value) {
                if (!$this->is_elementor_text_key((string) $key)) {
                    continue;
                }
                if (is_string($value)) {
                    $clean = trim(wp_strip_all_tags($value));
                    if ($clean !== '') {
                        $chunks[] = $clean;
                    }
                } elseif (is_array($value)) {
                    $this->collect_elementor_text($value, $chunks);
                }
            }
        }

        if (isset($node['elements']) && is_array($node['elements'])) {
            foreach ($node['elements'] as $child) {
                $this->collect_elementor_text($child, $chunks);
            }
        }
    }

    /**
     * Determine if an Elementor setting key is likely to contain text.
     */
    private function is_elementor_text_key(string $key): bool {
        $key = strtolower($key);
        $blocked = ['css', 'url', 'link', 'href', 'id', 'class', 'style'];
        foreach ($blocked as $needle) {
            if (strpos($key, $needle) !== false) {
                return false;
            }
        }

        $candidates = ['title', 'text', 'content', 'description', 'caption', 'heading', 'subheading', 'label', 'html', 'editor'];
        foreach ($candidates as $needle) {
            if (strpos($key, $needle) !== false) {
                return true;
            }
        }

        return false;
    }
    
    /**
     * Update meta data for a post
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function update_item($request) {
        return $this->try_callback(function() use ($request) {
            $post_id = (int) $request->get_param('post_id');
            
            // Verify post exists
            $post = get_post($post_id);
            if (!$post) {
                return $this->get_error_response(
                    __('Post not found.', 'prorank-seo'),
                    404,
                    'post_not_found'
                );
            }
            if (!current_user_can('edit_post', $post_id)) {
                return $this->get_error_response(
                    __('Insufficient permissions for this post.', 'prorank-seo'),
                    403,
                    'forbidden'
                );
            }
            
            $updated = [];
            
            // Update SEO title
            if ($request->has_param('seo_title')) {
                $seo_title = sanitize_text_field($request->get_param('seo_title'));
                $result = update_post_meta($post_id, '_prorank_seo_title', $seo_title);
                if ($result === false && $seo_title !== get_post_meta($post_id, '_prorank_seo_title', true)) {
                    return $this->get_error_response(
                        __('Failed to update SEO title.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['seo_title'] = $seo_title;
            }
            
            // Update meta description
            if ($request->has_param('seo_description')) {
                $seo_description = sanitize_textarea_field($request->get_param('seo_description'));
                $result = update_post_meta($post_id, '_prorank_seo_description', $seo_description);
                if ($result === false && $seo_description !== get_post_meta($post_id, '_prorank_seo_description', true)) {
                    return $this->get_error_response(
                        __('Failed to update meta description.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['seo_description'] = $seo_description;
            }
            
            // Update canonical URL
            if ($request->has_param('canonical_url')) {
                $canonical_url = esc_url_raw($request->get_param('canonical_url'));
                $result = update_post_meta($post_id, '_prorank_seo_canonical_url', $canonical_url);
                if ($result === false && $canonical_url !== get_post_meta($post_id, '_prorank_seo_canonical_url', true)) {
                    return $this->get_error_response(
                        __('Failed to update canonical URL.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['canonical_url'] = $canonical_url;
            }
            
            // Update meta robots
            if ($request->has_param('meta_robots')) {
                $meta_robots = $request->get_param('meta_robots');
                if (!is_array($meta_robots)) {
                    $meta_robots = [];
                }
                // Sanitize array values
                $meta_robots = array_map('sanitize_key', $meta_robots);
                $meta_robots = array_filter($meta_robots); // Remove empty values
                
                $result = update_post_meta($post_id, '_prorank_meta_robots', $meta_robots);
                if ($result === false) {
                    return $this->get_error_response(
                        __('Failed to update meta robots.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['meta_robots'] = $meta_robots;
            }
            
            // Update Open Graph title
            if ($request->has_param('og_title')) {
                $og_title = sanitize_text_field($request->get_param('og_title'));
                $result = update_post_meta($post_id, '_prorank_og_title', $og_title);
                if ($result === false && $og_title !== get_post_meta($post_id, '_prorank_og_title', true)) {
                    return $this->get_error_response(
                        __('Failed to update Open Graph title.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['og_title'] = $og_title;
            }
            
            // Update Open Graph description
            if ($request->has_param('og_description')) {
                $og_description = sanitize_textarea_field($request->get_param('og_description'));
                $result = update_post_meta($post_id, '_prorank_og_description', $og_description);
                if ($result === false && $og_description !== get_post_meta($post_id, '_prorank_og_description', true)) {
                    return $this->get_error_response(
                        __('Failed to update Open Graph description.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['og_description'] = $og_description;
            }
            
            // Update Open Graph image
            if ($request->has_param('og_image_id')) {
                $og_image_id = (int) $request->get_param('og_image_id');
                $result = update_post_meta($post_id, '_prorank_og_image_id', $og_image_id);
                if ($result === false) {
                    return $this->get_error_response(
                        __('Failed to update Open Graph image.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['og_image_id'] = $og_image_id;
            }
            
            // Update Twitter title
            if ($request->has_param('twitter_title')) {
                $twitter_title = sanitize_text_field($request->get_param('twitter_title'));
                $result = update_post_meta($post_id, '_prorank_twitter_title', $twitter_title);
                if ($result === false && $twitter_title !== get_post_meta($post_id, '_prorank_twitter_title', true)) {
                    return $this->get_error_response(
                        __('Failed to update Twitter title.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['twitter_title'] = $twitter_title;
            }
            
            // Update Twitter description
            if ($request->has_param('twitter_description')) {
                $twitter_description = sanitize_textarea_field($request->get_param('twitter_description'));
                $result = update_post_meta($post_id, '_prorank_twitter_description', $twitter_description);
                if ($result === false && $twitter_description !== get_post_meta($post_id, '_prorank_twitter_description', true)) {
                    return $this->get_error_response(
                        __('Failed to update Twitter description.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['twitter_description'] = $twitter_description;
            }
            
            // Update Twitter image
            if ($request->has_param('twitter_image_id')) {
                $twitter_image_id = (int) $request->get_param('twitter_image_id');
                $result = update_post_meta($post_id, '_prorank_twitter_image_id', $twitter_image_id);
                if ($result === false) {
                    return $this->get_error_response(
                        __('Failed to update Twitter image.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['twitter_image_id'] = $twitter_image_id;
            }
            
            // Update Twitter card type
            if ($request->has_param('twitter_card_type')) {
                $twitter_card_type = sanitize_key($request->get_param('twitter_card_type'));
                // Validate card type
                $valid_types = ['summary', 'summary_large_image', 'app', 'player'];
                if (!in_array($twitter_card_type, $valid_types, true)) {
                    $twitter_card_type = 'summary';
                }
                $result = update_post_meta($post_id, '_prorank_twitter_card_type', $twitter_card_type);
                if ($result === false && $twitter_card_type !== get_post_meta($post_id, '_prorank_twitter_card_type', true)) {
                    return $this->get_error_response(
                        __('Failed to update Twitter card type.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['twitter_card_type'] = $twitter_card_type;
            }
            
            // Update schema data
            if ($request->has_param('schema_data')) {
                $schema_data = $request->get_param('schema_data');
                if (!is_array($schema_data)) {
                    $schema_data = [];
                }
                
                // Sanitize schema data
                if (isset($schema_data['type'])) {
                    $schema_data['type'] = sanitize_text_field($schema_data['type']);
                }
                if (isset($schema_data['properties']) && is_array($schema_data['properties'])) {
                    // Recursively sanitize properties
                    $schema_data['properties'] = $this->sanitize_schema_properties($schema_data['properties']);
                }
                if (isset($schema_data['custom_json'])) {
                    // Validate JSON
                    if (!empty($schema_data['custom_json'])) {
                        $decoded = json_decode($schema_data['custom_json'], true);
                        if (json_last_error() !== JSON_ERROR_NONE) {
                            return $this->get_error_response(
                                __('Invalid JSON in custom schema.', 'prorank-seo'),
                                400,
                                'invalid_json'
                            );
                        }
                    }
                }
                
                $result = update_post_meta($post_id, '_prorank_schema_data', $schema_data);
                if ($result === false) {
                    return $this->get_error_response(
                        __('Failed to update schema data.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['schema_data'] = $schema_data;
            }

            // Update pillar content flag
            if ($request->has_param('is_pillar_content')) {
                $is_pillar = (bool) $request->get_param('is_pillar_content');
                update_post_meta($post_id, '_prorank_is_pillar_content', $is_pillar ? '1' : '');
                $updated['is_pillar_content'] = $is_pillar;
            }

            // Update focus keyword
            if ($request->has_param('focus_keyword')) {
                $focus_keyword = sanitize_text_field($request->get_param('focus_keyword'));
                $result = update_post_meta($post_id, '_prorank_focus_keyword', $focus_keyword);
                if ($result === false && $focus_keyword !== get_post_meta($post_id, '_prorank_focus_keyword', true)) {
                    return $this->get_error_response(
                        __('Failed to update focus keyword.', 'prorank-seo'),
                        500,
                        'update_failed'
                    );
                }
                $updated['focus_keyword'] = $focus_keyword;
            }

            return $this->get_success_response(
                $updated,
                __('Meta tags updated successfully.', 'prorank-seo')
            );
        });
    }
    
    /**
     * Sanitize schema properties recursively
     *
     * @param array $properties Schema properties to sanitize.
     * @return array
     */
    private function sanitize_schema_properties(array $properties): array {
        $sanitized = [];
        
        foreach ($properties as $key => $value) {
            $key = sanitize_key($key);
            
            if (is_array($value)) {
                $sanitized[$key] = $this->sanitize_schema_properties($value);
            } elseif (is_string($value)) {
                // Check if it's a URL
                if (filter_var($value, FILTER_VALIDATE_URL)) {
                    $sanitized[$key] = esc_url_raw($value);
                } else {
                    $sanitized[$key] = sanitize_textarea_field($value);
                }
            } elseif (is_numeric($value)) {
                $sanitized[$key] = $value;
            } elseif (is_bool($value)) {
                $sanitized[$key] = $value;
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Get meta fields schema
     *
     * @return array
     */
    private function get_meta_schema(): array {
        return [
            'seo_title' => [
                'description'       => __('SEO title for the post.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => false,
                'sanitize_callback' => 'sanitize_text_field',
                'maxLength'         => 60, // Recommended max length
            ],
            'seo_description' => [
                'description'       => __('Meta description for the post.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => false,
                'sanitize_callback' => 'sanitize_textarea_field',
                'maxLength'         => 160, // Recommended max length
            ],
            'canonical_url' => [
                'description'       => __('Canonical URL for the post.', 'prorank-seo'),
                'type'              => 'string',
                'format'            => 'uri',
                'required'          => false,
                'sanitize_callback' => 'esc_url_raw',
            ],
            'meta_robots' => [
                'description' => __('Meta robots directives.', 'prorank-seo'),
                'type'        => 'array',
                'required'    => false,
                'items'       => [
                    'type' => 'string',
                    'enum' => ['noindex', 'nofollow', 'noarchive', 'nosnippet', 'noimageindex'],
                ],
            ],
            'og_title' => [
                'description'       => __('Open Graph title for social sharing.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => false,
                'sanitize_callback' => 'sanitize_text_field',
                'maxLength'         => 80, // Facebook recommended
            ],
            'og_description' => [
                'description'       => __('Open Graph description for social sharing.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => false,
                'sanitize_callback' => 'sanitize_textarea_field',
                'maxLength'         => 200, // Facebook recommended
            ],
            'og_image_id' => [
                'description' => __('Open Graph image attachment ID.', 'prorank-seo'),
                'type'        => 'integer',
                'required'    => false,
                'minimum'     => 0,
            ],
            'twitter_title' => [
                'description'       => __('Twitter title for social sharing.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => false,
                'sanitize_callback' => 'sanitize_text_field',
                'maxLength'         => 70, // Twitter recommended
            ],
            'twitter_description' => [
                'description'       => __('Twitter description for social sharing.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => false,
                'sanitize_callback' => 'sanitize_textarea_field',
                'maxLength'         => 200, // Twitter recommended
            ],
            'twitter_image_id' => [
                'description' => __('Twitter image attachment ID.', 'prorank-seo'),
                'type'        => 'integer',
                'required'    => false,
                'minimum'     => 0,
            ],
            'twitter_card_type' => [
                'description' => __('Twitter card type.', 'prorank-seo'),
                'type'        => 'string',
                'required'    => false,
                'enum'        => ['summary', 'summary_large_image', 'app', 'player'],
                'default'     => 'summary',
            ],
            'schema_data' => [
                'description' => __('Schema.org structured data configuration.', 'prorank-seo'),
                'type'        => 'object',
                'required'    => false,
                'properties'  => [
                    'type' => [
                        'type' => 'string',
                        'description' => __('Primary schema type.', 'prorank-seo'),
                    ],
                    'properties' => [
                        'type' => 'object',
                        'description' => __('Schema type-specific properties.', 'prorank-seo'),
                    ],
                    'custom_json' => [
                        'type' => 'string',
                        'description' => __('Custom JSON-LD markup.', 'prorank-seo'),
                    ],
                ],
            ],
            'is_pillar_content' => [
                'description' => __('Mark this content as pillar/cornerstone content for internal linking.', 'prorank-seo'),
                'type'        => 'boolean',
                'required'    => false,
                'default'     => false,
            ],
            'focus_keyword' => [
                'description'       => __('Focus keyword for SEO analysis.', 'prorank-seo'),
                'type'              => 'string',
                'required'          => false,
                'sanitize_callback' => 'sanitize_text_field',
                'maxLength'         => 100,
            ],
        ];
    }

    /**
     * Get link statistics for a post
     *
     * @param int    $post_id Post ID.
     * @param string $content Post content.
     * @return array Link statistics.
     */
    private function get_post_link_stats(int $post_id, string $content): array {
        // Count internal and external links
        $internal_links = 0;
        $external_links = 0;
        $anchor_texts = [];
        $site_url = home_url();

        // Use DOMDocument for reliable parsing
        if (!empty($content)) {
            $dom = new \DOMDocument();
            libxml_use_internal_errors(true);
            $dom->loadHTML('<?xml encoding="UTF-8">' . $content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
            libxml_clear_errors();

            $links = $dom->getElementsByTagName('a');
            foreach ($links as $link) {
                $href = $link->getAttribute('href');
                $anchor = trim($link->textContent);

                if (empty($href) || strpos($href, '#') === 0 || strpos($href, 'mailto:') === 0) {
                    continue;
                }

                // Check if internal or external
                if (strpos($href, $site_url) === 0 || strpos($href, '/') === 0) {
                    $internal_links++;
                    if (!empty($anchor)) {
                        $anchor_texts[] = $anchor;
                    }
                } else {
                    $external_links++;
                }
            }
        }

        $total_links = $internal_links + $external_links;

        // Calculate anchor text diversity (unique anchors / total internal)
        $unique_anchors = count(array_unique($anchor_texts));
        $anchor_diversity = $internal_links > 0 ? round(($unique_anchors / $internal_links) * 100) : 100;

        // Warnings
        $warnings = [];
        if ($total_links > 150) {
            $warnings[] = __('Too many links on this page (150+ may dilute link equity).', 'prorank-seo');
        } elseif ($total_links > 100) {
            $warnings[] = __('High link count. Consider reducing to under 100 links.', 'prorank-seo');
        }

        if ($anchor_diversity < 50 && $internal_links > 3) {
            $warnings[] = __('Low anchor text diversity. Vary your anchor text for better SEO.', 'prorank-seo');
        }

        return [
            'internal_links' => $internal_links,
            'external_links' => $external_links,
            'total_links' => $total_links,
            'anchor_diversity' => $anchor_diversity,
            'warnings' => $warnings,
        ];
    }
}
