<?php
/**
 * Feedback REST API Controller (Free)
 *
 * Stores feedback submissions locally for admins.
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

class FeedbackController extends BaseController {
    /**
     * Constructor
     */
    public function __construct() {
        $this->rest_base = 'feedback';
    }

    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/submit',
            [
                [
                    'methods'             => \WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'submit_feedback'],
                    'permission_callback' => [$this, 'submit_feedback_permissions_check'],
                    'args'                => $this->get_comprehensive_feedback_schema(),
                ],
            ]
        );

        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/list',
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_feedback_list'],
                    'permission_callback' => [$this, 'submit_feedback_permissions_check'],
                    'args'                => [
                        'type' => [
                            'description' => __('Filter by feedback type', 'prorank-seo'),
                            'type'        => 'string',
                            'enum'        => ['all', 'feature', 'bug', 'general', 'support'],
                            'default'     => 'all',
                        ],
                        'limit' => [
                            'description' => __('Number of items to return', 'prorank-seo'),
                            'type'        => 'integer',
                            'default'     => 10,
                            'minimum'     => 1,
                            'maximum'     => 100,
                        ],
                    ],
                ],
            ]
        );
    }

    /**
     * Check permissions for submitting feedback
     *
     * @param WP_REST_Request $request Request object.
     * @return bool|WP_Error
     */
    public function submit_feedback_permissions_check(WP_REST_Request $request) {
        if (!current_user_can('manage_options')) {
            return new WP_Error(
                'rest_forbidden',
                __('You do not have permission to submit feedback.', 'prorank-seo'),
                ['status' => 403]
            );
        }

        return true;
    }

    /**
     * Submit feedback (stored locally)
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function submit_feedback(WP_REST_Request $request) {
        return $this->try_callback(function() use ($request) {
            $type = $request->get_param('type');
            $subject = $request->get_param('subject');
            $message = $request->get_param('message');
            $priority = $request->get_param('priority');
            $email = $request->get_param('email');

            $current_user = wp_get_current_user();
            $user_email = $email ?: $current_user->user_email;
            $user_name = $current_user->display_name ?: __('Anonymous', 'prorank-seo');

            $feedback_data = [
                'id' => wp_generate_uuid4(),
                'type' => $type,
                'subject' => $subject,
                'message' => $message,
                'priority' => $priority ?: 'medium',
                'email' => $user_email,
                'name' => $user_name,
                'site_url' => get_site_url(),
                'submitted_at' => current_time('mysql'),
                'status' => 'submitted',
            ];

            $this->store_feedback($feedback_data);

            return $this->get_success_response(
                ['id' => $feedback_data['id']],
                __('Thank you for your feedback! We\'ll review it soon.', 'prorank-seo')
            );
        });
    }

    /**
     * Get feedback list (local)
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function get_feedback_list(WP_REST_Request $request) {
        return $this->try_callback(function() use ($request) {
            $type = $request->get_param('type');
            $limit = (int) $request->get_param('limit');

            $feedback_queue = get_option('prorank_feedback_queue', []);
            if (!is_array($feedback_queue)) {
                $feedback_queue = [];
            }

            if ($type && $type !== 'all') {
                $feedback_queue = array_filter($feedback_queue, function($item) use ($type) {
                    return isset($item['type']) && $item['type'] === $type;
                });
            }

            usort($feedback_queue, function($a, $b) {
                $time_a = $a['submitted_at'] ?? '1970-01-01 00:00:00';
                $time_b = $b['submitted_at'] ?? '1970-01-01 00:00:00';
                return strtotime($time_b) - strtotime($time_a);
            });

            $feedback_queue = array_slice($feedback_queue, 0, max(1, $limit));

            return $this->get_success_response([
                'feedback' => array_values($feedback_queue),
                'stats' => $this->build_feedback_stats($feedback_queue),
                'source' => 'local',
            ]);
        });
    }

    /**
     * Store feedback locally
     *
     * @param array $feedback_data Feedback data.
     * @return void
     */
    private function store_feedback(array $feedback_data): void {
        $feedback_queue = get_option('prorank_feedback_queue', []);
        if (!is_array($feedback_queue)) {
            $feedback_queue = [];
        }

        $feedback_queue[] = $feedback_data;
        if (count($feedback_queue) > 100) {
            $feedback_queue = array_slice($feedback_queue, -100);
        }

        update_option('prorank_feedback_queue', $feedback_queue);
    }

    /**
     * Build feedback stats summary
     *
     * @param array $feedback_queue Feedback list.
     * @return array
     */
    private function build_feedback_stats(array $feedback_queue): array {
        $stats = [
            'feature' => 0,
            'bug' => 0,
            'general' => 0,
            'support' => 0,
        ];

        foreach ($feedback_queue as $item) {
            $type = $item['type'] ?? '';
            if (isset($stats[$type])) {
                $stats[$type]++;
            }
        }

        return $stats;
    }

    /**
     * Get comprehensive feedback schema
     *
     * @return array
     */
    private function get_comprehensive_feedback_schema(): array {
        return [
            'type' => [
                'description'       => __('Type of feedback', 'prorank-seo'),
                'type'              => 'string',
                'enum'              => ['feature', 'bug', 'general', 'support'],
                'required'          => true,
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'subject' => [
                'description'       => __('Feedback subject', 'prorank-seo'),
                'type'              => 'string',
                'required'          => true,
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'message' => [
                'description'       => __('Feedback message', 'prorank-seo'),
                'type'              => 'string',
                'required'          => true,
                'sanitize_callback' => 'sanitize_textarea_field',
            ],
            'priority' => [
                'description'       => __('Priority level', 'prorank-seo'),
                'type'              => 'string',
                'enum'              => ['low', 'medium', 'high'],
                'default'           => 'medium',
                'sanitize_callback' => 'sanitize_text_field',
            ],
            'email' => [
                'description'       => __('Contact email', 'prorank-seo'),
                'type'              => 'string',
                'format'            => 'email',
                'sanitize_callback' => 'sanitize_email',
            ],
        ];
    }
}
