<?php
/**
 * Enhanced Audit REST API Controller (Free)
 *
 * Local-only audit endpoints for the free plugin.
 *
 * @package ProRank\SEO\Core\RestApi
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use ProRank\SEO\Core\Audits\BasicAuditEngine;
use ProRank\SEO\Core\AuditFixHandler;

class EnhancedAuditController extends BaseController {
    /**
     * Register routes
     */
    public function register_routes(): void {
        // Audit status
        register_rest_route($this->namespace, '/audit/status', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_enhanced_status'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);

        // Start audit
        register_rest_route($this->namespace, '/audit/start', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'start_deep_audit'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'type' => [
                        'required'          => false,
                        'default'           => 'basic',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Background audit processor (internal async)
        register_rest_route($this->namespace, '/audit/process', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'process_deep_audit'],
                'permission_callback' => [$this, 'check_process_token'],
                'args'                => [
                    'audit_id' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'token' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Audit issues
        register_rest_route($this->namespace, '/audit/issues', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_audit_issues'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);

        // Audit history
        register_rest_route($this->namespace, '/audit/history', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_audit_history'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'limit' => [
                        'required'          => false,
                        'default'           => 10,
                        'sanitize_callback' => 'absint',
                    ],
                ],
            ],
        ]);

        // Audit report details
        register_rest_route($this->namespace, '/audit/report/(?P<id>[\w-]+)', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'get_audit_report'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'id' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Stop audit (local audits run synchronously, but keep endpoint for UI)
        register_rest_route($this->namespace, '/audit/stop', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'stop_audit'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);

        // Quick fix endpoint (local processing only in free plugin)
        register_rest_route($this->namespace, '/audit/quick-fix', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'apply_quick_fix'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'fix_id' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'fix_type' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ]);

        // Fix broken link endpoint (local processing only in free plugin)
        register_rest_route($this->namespace, '/audit/fix-broken-link', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [$this, 'fix_broken_link'],
                'permission_callback' => [$this, 'check_permission'],
                'args'                => [
                    'link_id' => [
                        'required'          => true,
                        'sanitize_callback' => 'absint',
                    ],
                    'action' => [
                        'required'          => true,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'new_url' => [
                        'required'          => false,
                        'sanitize_callback' => 'esc_url_raw',
                    ],
                ],
            ],
        ]);

        // Ping endpoint for UI connection checks
        register_rest_route($this->namespace, '/audit/ping', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [$this, 'ping_audit_service'],
                'permission_callback' => [$this, 'check_permission'],
            ],
        ]);
    }

    /**
     * Get audit status (local basic)
     */
    public function get_enhanced_status(WP_REST_Request $request): WP_REST_Response {
        $engine = new BasicAuditEngine();
        $status = $engine->get_latest_status();

        if (empty($status)) {
            return new WP_REST_Response([
                'state' => 'idle',
                'message' => __('No active audit', 'prorank-seo'),
                'can_start' => true,
            ], 200);
        }

        $status['score'] = $status['overallScore'] ?? $status['overall_score'] ?? null;
        $status['stats'] = $this->build_stats_from_status($status);
        $status['quick_fixes'] = [];

        return new WP_REST_Response($status, 200);
    }

    /**
     * Start a local basic audit
     */
    public function start_deep_audit(WP_REST_Request $request): WP_REST_Response {
        // Get settings from request - frontend sends { mode: 'local', settings: {...} }
        $settings = $request->get_param('settings');

        // If settings param not found, try extracting from JSON body
        if (!is_array($settings)) {
            $json_params = $request->get_json_params();
            if (is_array($json_params) && isset($json_params['settings']) && is_array($json_params['settings'])) {
                $settings = $json_params['settings'];
            } else {
                $settings = $json_params;
            }
        }

        // Fallback to saved settings if still not an array
        if (!is_array($settings)) {
            $settings = get_option('prorank_audit_settings', []);
        }

        $engine = new BasicAuditEngine();
        $start = $engine->start_audit($settings);

        if (is_wp_error($start)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $start->get_error_message(),
                'error' => $start->get_error_code(),
            ], 400);
        }

        $audit_id = (string) ($start['audit_id'] ?? '');
        $token = wp_generate_password(32, false);
        set_transient('prorank_audit_token_' . $audit_id, $token, 10 * MINUTE_IN_SECONDS);

        $process_url = rest_url($this->namespace . '/audit/process');
        $async = wp_remote_post($process_url, [
            'timeout' => 0.01,
            'blocking' => false,
            'body' => [
                'audit_id' => $audit_id,
                'token' => $token,
            ],
        ]);

        if (is_wp_error($async)) {
            $result = $engine->process_audit($audit_id);
            if (is_wp_error($result)) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => $result->get_error_message(),
                    'error' => $result->get_error_code(),
                ], 400);
            }

            return new WP_REST_Response([
                'success' => true,
                'message' => __('Basic audit completed locally', 'prorank-seo'),
                'audit_id' => $result['audit_id'] ?? null,
                'status' => 'completed',
                'mode' => 'local_basic',
            ], 200);
        }

        return new WP_REST_Response([
            'success' => true,
            'message' => __('Audit started', 'prorank-seo'),
            'audit_id' => $audit_id,
            'status' => 'checking',
            'state' => 'checking',
            'progress' => $start['progress'] ?? [],
            'mode' => 'local_basic',
        ], 202);
    }

    /**
     * Validate background audit token.
     */
    public function check_process_token(WP_REST_Request $request): bool {
        if (current_user_can('manage_options')) {
            return true;
        }

        $audit_id = sanitize_text_field((string) $request->get_param('audit_id'));
        $token = sanitize_text_field((string) $request->get_param('token'));

        if ($audit_id === '' || $token === '') {
            return false;
        }

        $stored = get_transient('prorank_audit_token_' . $audit_id);
        if (!is_string($stored) || $stored === '') {
            return false;
        }

        return hash_equals($stored, $token);
    }

    /**
     * Process audit asynchronously.
     */
    public function process_deep_audit(WP_REST_Request $request): WP_REST_Response {
        $audit_id = sanitize_text_field((string) $request->get_param('audit_id'));

        if ($audit_id === '') {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Audit ID is required', 'prorank-seo'),
            ], 400);
        }

        $engine = new BasicAuditEngine();
        $result = $engine->process_audit($audit_id);
        delete_transient('prorank_audit_token_' . $audit_id);

        if (is_wp_error($result)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => $result->get_error_message(),
                'error' => $result->get_error_code(),
            ], 400);
        }

        return new WP_REST_Response([
            'success' => true,
            'audit_id' => $audit_id,
            'status' => 'completed',
        ], 200);
    }

    /**
     * Get audit history
     */
    public function get_audit_history(WP_REST_Request $request): WP_REST_Response {
        $limit = $request->get_param('limit');
        $limit = is_numeric($limit) ? (int) $limit : 10;
        $limit = max(1, min(50, $limit));

        $engine = new BasicAuditEngine();
        $audits = $engine->get_history($limit);

        $normalized = array_map(function ($audit) {
            $audit_id = $audit['audit_id'] ?? $audit['id'] ?? '';
            $score = $audit['overallScore'] ?? $audit['overall_score'] ?? 0;
            $counts = $audit['issues'] ?? [];

            return [
                'id' => $audit_id,
                'audit_id' => $audit_id,
                'date' => $audit['completed_at'] ?? $audit['started_at'] ?? $audit['created_at'] ?? current_time('mysql'),
                'score' => (int) $score,
                'totalUrls' => (int) ($audit['total_urls'] ?? $audit['totalUrls'] ?? 0),
                'total_issues' => (int) ($counts['total'] ?? 0),
                'critical' => (int) ($counts['critical'] ?? 0),
                'duration' => null,
                'stats' => [
                    'total_urls' => (int) ($audit['total_urls'] ?? $audit['totalUrls'] ?? 0),
                    'total_issues' => (int) ($counts['total'] ?? 0),
                    'critical_issues' => (int) ($counts['critical'] ?? 0),
                    'high_issues' => (int) ($counts['high'] ?? 0),
                    'medium_issues' => (int) ($counts['medium'] ?? 0),
                    'low_issues' => (int) ($counts['low'] ?? 0),
                ],
                'completed_at' => $audit['completed_at'] ?? null,
                'started_at' => $audit['started_at'] ?? null,
            ];
        }, $audits);

        return new WP_REST_Response([
            'audits' => $normalized,
            'total' => count($normalized),
        ], 200);
    }

    /**
     * Get audit issues
     */
    public function get_audit_issues(WP_REST_Request $request): WP_REST_Response {
        $engine = new BasicAuditEngine();
        $status = $engine->get_latest_status();

        if (empty($status['audit_id'])) {
            return new WP_REST_Response(['issues' => []], 200);
        }

        $issues = $engine->get_issues((string) $status['audit_id']);

        return new WP_REST_Response([
            'issues' => $issues,
        ], 200);
    }

    /**
     * Get audit report details
     */
    public function get_audit_report(WP_REST_Request $request): WP_REST_Response {
        $report_id = (string) $request->get_param('id');

        if ($report_id === '') {
            return new WP_REST_Response([
                'error' => __('Report ID is required', 'prorank-seo'),
            ], 400);
        }

        $report = $this->build_report_data($report_id);

        if (is_wp_error($report)) {
            return new WP_REST_Response([
                'error' => $report->get_error_message(),
            ], (int) ($report->get_error_data('status') ?? 404));
        }

        return new WP_REST_Response($report, 200);
    }

    /**
     * Download audit report as HTML via admin-ajax.
     */
    public function download_audit_report_html(): void {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Sorry, you are not allowed to do that.', 'prorank-seo'), '', ['response' => 403]);
        }

        check_ajax_referer('wp_rest', '_wpnonce');

        $report_id = isset($_GET['report_id']) ? sanitize_text_field(wp_unslash((string) $_GET['report_id'])) : '';
        if ($report_id === '') {
            wp_die(esc_html__('Report ID is required', 'prorank-seo'), '', ['response' => 400]);
        }

        $report = $this->build_report_data($report_id);
        if (is_wp_error($report)) {
            wp_die(esc_html($report->get_error_message()), '', ['response' => (int) ($report->get_error_data('status') ?? 404)]);
        }

        $html = $this->render_report_html($report);
        $filename = sprintf('prorank-audit-%s.html', sanitize_file_name($report_id));

        nocache_headers();
        header('Content-Type: text/html; charset=' . get_option('blog_charset'));
        header('Content-Disposition: inline; filename="' . $filename . '"');
        header('X-Content-Type-Options: nosniff');

        echo wp_kses($html, $this->get_report_html_allowed_tags());
        exit;
    }

    /**
     * Allowed HTML tags for audit report output.
     */
    private function get_report_html_allowed_tags(): array {
        return [
            'html' => ['lang' => true],
            'head' => [],
            'meta' => ['charset' => true],
            'title' => [],
            'style' => [],
            'body' => [],
            'h1' => [],
            'h2' => [],
            'h3' => [],
            'div' => ['class' => true],
            'span' => ['class' => true],
            'strong' => [],
            'table' => [],
            'thead' => [],
            'tbody' => [],
            'tr' => [],
            'th' => [],
            'td' => ['colspan' => true],
            'ul' => ['class' => true],
            'li' => [],
            'a' => ['href' => true, 'target' => true, 'rel' => true],
            'p' => [],
            'br' => [],
        ];
    }

    /**
     * Build report data for local audits.
     *
     * @param string $report_id Report ID.
     * @return array|\WP_Error
     */
    private function build_report_data(string $report_id) {
        $engine = new BasicAuditEngine();
        $issues = $engine->get_issues($report_id);

        if (empty($issues)) {
            return new \WP_Error('report_not_found', __('Report not found', 'prorank-seo'), ['status' => 404]);
        }

        $counts = $this->count_issues($issues);
        $issues_by_page = $this->group_issues_by_page($issues);
        $category_breakdown = $this->build_category_breakdown($issues);

        return [
            'id' => $report_id,
            'date' => current_time('mysql'),
            'score' => $this->calculate_score($counts),
            'stats' => [
                'critical_issues' => $counts['critical'],
                'high_issues' => $counts['high'],
                'medium_issues' => $counts['medium'],
                'low_issues' => $counts['low'],
                'total_issues' => $counts['total'],
                'total_urls' => count($issues_by_page),
            ],
            'issues_by_page' => $issues_by_page,
            'category_breakdown' => $category_breakdown,
        ];
    }

    /**
     * Render audit report HTML.
     *
     * @param array $report Report data.
     * @return string
     */
    private function render_report_html(array $report): string {
        $report_id = esc_html($report['id'] ?? '');
        $date = esc_html($report['date'] ?? '');
        $score = (int) ($report['score'] ?? 0);
        $stats = is_array($report['stats'] ?? null) ? $report['stats'] : [];
        $issues_by_page = is_array($report['issues_by_page'] ?? null) ? $report['issues_by_page'] : [];
        $category_breakdown = is_array($report['category_breakdown'] ?? null) ? $report['category_breakdown'] : [];

        $critical = (int) ($stats['critical_issues'] ?? 0);
        $high = (int) ($stats['high_issues'] ?? 0);
        $medium = (int) ($stats['medium_issues'] ?? 0);
        $low = (int) ($stats['low_issues'] ?? 0);
        $total = (int) ($stats['total_issues'] ?? 0);
        $total_urls = (int) ($stats['total_urls'] ?? 0);

        $score_class = $score >= 80 ? 'score-good' : ($score >= 60 ? 'score-ok' : 'score-poor');

        wp_enqueue_style(
            'prorank-audit-report',
            plugins_url('assets/css/audit-report.css', PRORANK_PLUGIN_FILE),
            [],
            PRORANK_VERSION
        );

        ob_start();
        ?>
        <!doctype html>
        <html lang="en">
        <head>
            <meta charset="<?php echo esc_attr(get_option('blog_charset')); ?>">
            <title><?php echo esc_html__('ProRank SEO Audit Report', 'prorank-seo'); ?></title>
            <?php wp_print_styles('prorank-audit-report'); ?>
        </head>
        <body>
            <h1><?php echo esc_html__('ProRank SEO Audit Report', 'prorank-seo'); ?></h1>
            <div class="meta">
                <?php echo esc_html__('Report ID:', 'prorank-seo'); ?> <?php echo esc_html($report_id); ?> ·
                <?php echo esc_html__('Date:', 'prorank-seo'); ?> <?php echo esc_html($date); ?>
            </div>

            <div class="score">
                <div class="score-badge <?php echo esc_attr($score_class); ?>"><?php echo esc_html((string) $score); ?></div>
                <div>
                    <div><strong><?php echo esc_html__('Overall Score', 'prorank-seo'); ?></strong></div>
                    <div class="meta"><?php echo esc_html__('Higher is better', 'prorank-seo'); ?></div>
                </div>
            </div>

            <div class="stats">
                <div class="stat"><strong><?php echo esc_html((string) $total); ?></strong><?php echo esc_html__('Total issues', 'prorank-seo'); ?></div>
                <div class="stat"><strong><?php echo esc_html((string) $critical); ?></strong><?php echo esc_html__('Critical', 'prorank-seo'); ?></div>
                <div class="stat"><strong><?php echo esc_html((string) $high); ?></strong><?php echo esc_html__('High', 'prorank-seo'); ?></div>
                <div class="stat"><strong><?php echo esc_html((string) $medium); ?></strong><?php echo esc_html__('Medium', 'prorank-seo'); ?></div>
                <div class="stat"><strong><?php echo esc_html((string) $low); ?></strong><?php echo esc_html__('Low', 'prorank-seo'); ?></div>
                <div class="stat"><strong><?php echo esc_html((string) $total_urls); ?></strong><?php echo esc_html__('Pages scanned', 'prorank-seo'); ?></div>
            </div>

            <h2><?php echo esc_html__('Category Breakdown', 'prorank-seo'); ?></h2>
            <table>
                <thead>
                    <tr>
                        <th><?php echo esc_html__('Category', 'prorank-seo'); ?></th>
                        <th><?php echo esc_html__('Issues', 'prorank-seo'); ?></th>
                        <th><?php echo esc_html__('Score', 'prorank-seo'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($category_breakdown)) : ?>
                        <tr><td colspan="3"><?php echo esc_html__('No category data available.', 'prorank-seo'); ?></td></tr>
                    <?php else : ?>
                        <?php foreach ($category_breakdown as $category => $data) : ?>
                            <tr>
                                <td><?php echo esc_html((string) $category); ?></td>
                                <td><?php echo esc_html((string) ($data['issues'] ?? 0)); ?></td>
                                <td><?php echo esc_html((string) ($data['score'] ?? 0)); ?>%</td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>

            <h2><?php echo esc_html__('Issues by Page', 'prorank-seo'); ?></h2>
            <?php if (empty($issues_by_page)) : ?>
                <p><?php echo esc_html__('No issues found for this report.', 'prorank-seo'); ?></p>
            <?php else : ?>
                <?php foreach ($issues_by_page as $url => $issues) : ?>
                    <h3><a href="<?php echo esc_url((string) $url); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html((string) $url); ?></a></h3>
                    <ul class="issue-list">
                        <?php foreach ($issues as $issue) : ?>
                            <?php
                                $severity = strtolower((string) ($issue['severity'] ?? 'low'));
                                $severity_class = 'severity-' . $severity;
                                $check_type = (string) ($issue['check_type'] ?? '');
                                $message = (string) ($issue['message'] ?? '');
                            ?>
                            <li>
                                <span class="<?php echo esc_attr($severity_class); ?>"><?php echo esc_html(ucfirst($severity)); ?></span>
                                <?php if ($check_type !== '') : ?>
                                    — <?php echo esc_html($check_type); ?>
                                <?php endif; ?>
                                <?php if ($message !== '') : ?>
                                    : <?php echo esc_html($message); ?>
                                <?php endif; ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endforeach; ?>
            <?php endif; ?>
        </body>
        </html>
        <?php
        return (string) ob_get_clean();
    }

    /**
     * Stop audit (no-op for synchronous local audits)
     */
    public function stop_audit(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response([
            'success' => true,
            'message' => __('No active audit to stop', 'prorank-seo'),
        ], 200);
    }

    /**
     * Apply a local quick fix using the built-in fix handler.
     */
    public function apply_quick_fix(WP_REST_Request $request): WP_REST_Response {
        $fix_type = sanitize_key(str_replace('-', '_', (string) $request->get_param('fix_type')));

        if ($fix_type === '') {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Fix type is required.', 'prorank-seo'),
            ], 400);
        }

        $payload = $request->get_json_params();
        if (!is_array($payload)) {
            $payload = $request->get_params();
        }

        $payload = $this->sanitize_fix_payload($payload);
        $payload['fix_type'] = $fix_type;
        $payload['fix_id'] = sanitize_text_field((string) $request->get_param('fix_id'));

        $handler = new AuditFixHandler();
        $result = $handler->apply_fix($fix_type, $payload);

        if (!is_array($result)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Unable to process the requested fix.', 'prorank-seo'),
            ], 500);
        }

        if (!array_key_exists('success', $result)) {
            $result['success'] = false;
        }

        return new WP_REST_Response($result, 200);
    }

    /**
     * Fix broken links locally via the internal-linking endpoint handler.
     */
    public function fix_broken_link(WP_REST_Request $request): WP_REST_Response {
        if (!function_exists(__NAMESPACE__ . '\fix_broken_link')) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Broken link fixer is unavailable.', 'prorank-seo'),
            ], 500);
        }

        $link_id = absint($request->get_param('link_id'));
        if ($link_id <= 0) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('Valid link ID is required.', 'prorank-seo'),
            ], 400);
        }

        $action = sanitize_key((string) $request->get_param('action'));
        if ($action === 'update') {
            $action = 'replace';
        }
        if (!in_array($action, ['replace', 'remove', 'unlink'], true)) {
            $action = 'remove';
        }

        $proxy_request = new WP_REST_Request('POST', '/prorank-seo/v1/linking/fix-broken-link');
        $proxy_request->set_header('X-WP-Nonce', (string) $request->get_header('X-WP-Nonce'));
        $proxy_request->set_param('id', $link_id);
        $proxy_request->set_param('action', $action);

        if ($action === 'replace') {
            $new_url = esc_url_raw((string) $request->get_param('new_url'));
            if ($new_url === '') {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => __('A valid replacement URL is required.', 'prorank-seo'),
                ], 400);
            }
            $proxy_request->set_param('new_url', $new_url);
        }

        return \ProRank\SEO\Core\RestApi\fix_broken_link($proxy_request);
    }

    /**
     * Ping endpoint
     */
    public function ping_audit_service(WP_REST_Request $request): WP_REST_Response {
        return new WP_REST_Response([
            'success' => true,
            'message' => 'pong',
            'timestamp' => time(),
        ], 200);
    }

    private function build_stats_from_status(array $status): array {
        $critical = (int) ($status['issuesCritical'] ?? 0);
        $high = (int) ($status['issuesHigh'] ?? 0);
        $medium = (int) ($status['issuesMedium'] ?? 0);
        $low = (int) ($status['issuesLow'] ?? 0);
        $total = $critical + $high + $medium + $low;

        return [
            'critical_issues' => $critical,
            'high_issues' => $high,
            'medium_issues' => $medium,
            'low_issues' => $low,
            'total_issues' => $total,
            'total_urls' => (int) ($status['total_urls'] ?? $status['totalUrls'] ?? 0),
        ];
    }

    private function count_issues(array $issues): array {
        $counts = [
            'critical' => 0,
            'high' => 0,
            'medium' => 0,
            'low' => 0,
            'total' => 0,
        ];

        foreach ($issues as $issue) {
            $severity = (string) ($issue['severity'] ?? 'low');
            if (!isset($counts[$severity])) {
                $severity = 'low';
            }
            $counts[$severity]++;
            $counts['total']++;
        }

        return $counts;
    }

    private function calculate_score(array $counts): int {
        $score = 100;
        $score -= min(60, $counts['critical'] * 15);
        $score -= min(40, $counts['high'] * 8);
        $score -= min(25, $counts['medium'] * 4);
        $score -= min(15, $counts['low'] * 2);

        return max(0, min(100, $score));
    }

    private function group_issues_by_page(array $issues): array {
        $grouped = [];

        foreach ($issues as $issue) {
            $url = (string) ($issue['url'] ?? '');
            if ($url === '') {
                $url = home_url('/');
            }

            if (!isset($grouped[$url])) {
                $grouped[$url] = [];
            }

            $grouped[$url][] = [
                'severity' => $issue['severity'] ?? 'low',
                'check_type' => $issue['type'] ?? $issue['issue_type'] ?? __('Issue', 'prorank-seo'),
                'message' => $issue['message'] ?? $issue['description'] ?? '',
            ];
        }

        return $grouped;
    }

    private function build_category_breakdown(array $issues): array {
        $categories = [];

        foreach ($issues as $issue) {
            $category = (string) ($issue['display_category'] ?? $issue['category'] ?? 'Other');
            if ($category === '') {
                $category = 'Other';
            }

            if (!isset($categories[$category])) {
                $categories[$category] = [
                    'issues' => 0,
                    'score' => 100,
                ];
            }

            $categories[$category]['issues']++;
        }

        foreach ($categories as $key => $data) {
            $score = max(0, 100 - ($data['issues'] * 10));
            $categories[$key]['score'] = $score;
        }

        return $categories;
    }

    /**
     * Sanitize quick-fix payload values recursively.
     *
     * @param mixed $value Input value.
     * @return mixed
     */
    private function sanitize_fix_payload($value) {
        if (is_array($value)) {
            $clean = [];
            foreach ($value as $key => $item) {
                $clean_key = is_string($key) ? sanitize_key($key) : $key;
                $clean[$clean_key] = $this->sanitize_fix_payload($item);
            }
            return $clean;
        }

        if (is_bool($value) || is_int($value) || is_float($value) || $value === null) {
            return $value;
        }

        return sanitize_text_field((string) $value);
    }
}
