<?php
/**
 * Database Optimization REST API Controller
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * DatabaseOptimizationController class
 */
class DatabaseOptimizationController extends \WP_REST_Controller {
    
    /**
     * Module instance
     *
     * @var \ProRank\SEO\Modules\Performance\DatabaseOptimizationModule|null
     */
    private $module = null;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->namespace = 'prorank-seo/v1';
        $this->rest_base = 'database-optimization';
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Analyze database
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/analyze',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'analyze_database'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Analyze autoload
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/analyze-autoload',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'analyze_autoload'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Get database statistics
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/stats',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_database_stats'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Get table overhead
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/overhead',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_table_overhead'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Cleanup database
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/cleanup',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'cleanup_database'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args' => [
                        'type' => [
                            'required' => true,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                        'retention_days' => [
                            'required' => false,
                            'type' => 'integer',
                            'default' => 0,
                        ],
                    ],
                ],
            ]
        );
        
        // Get database overview
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/overview',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_database_overview'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Get all tables info
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/tables',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_tables_info'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Table operations
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/table/(?P<action>repair|empty|optimize|convert)',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'table_operation'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args' => [
                        'action' => [
                            'required' => true,
                            'type' => 'string',
                            'enum' => ['repair', 'empty', 'optimize', 'convert'],
                        ],
                        'table_name' => [
                            'required' => false,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                        'table' => [
                            'required' => false,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                    ],
                ],
            ]
        );
        
        // Get all options
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/options',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_options'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                    'args' => [
                        'search' => [
                            'required' => false,
                            'type' => 'string',
                            'default' => '',
                        ],
                        'autoload' => [
                            'required' => false,
                            'type' => 'string',
                            'default' => '',
                        ],
                        'limit' => [
                            'required' => false,
                            'type' => 'integer',
                            'default' => 100,
                        ],
                        'offset' => [
                            'required' => false,
                            'type' => 'integer',
                            'default' => 0,
                        ],
                    ],
                ],
            ]
        );
        
        // Update option autoload
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/option/autoload',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'update_option_autoload'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args' => [
                        'option_name' => [
                            'required' => true,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                        'autoload' => [
                            'required' => true,
                            'type' => 'string',
                            'enum' => ['yes', 'no'],
                        ],
                    ],
                ],
            ]
        );
        
        // Get cron jobs
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/cron',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_cron_jobs'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Delete cron job
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/cron/delete',
            [
                [
                    'methods'             => WP_REST_Server::DELETABLE,
                    'callback'            => [$this, 'delete_cron_job'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args' => [
                        'hook' => [
                            'required' => true,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                        'timestamp' => [
                            'required' => true,
                            'type' => 'integer',
                        ],
                    ],
                ],
            ]
        );
        
        // Get orphaned items preview
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/orphaned/(?P<type>\w+)',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_orphaned_preview'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                    'args' => [
                        'type' => [
                            'required' => true,
                            'type' => 'string',
                            'enum' => ['postmeta', 'commentmeta', 'usermeta', 'termmeta', 'relationships'],
                        ],
                    ],
                ],
            ]
        );
        
        // Create backup
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/backup/create',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'create_backup'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args' => [
                        'tables' => [
                            'required' => false,
                            'type' => 'array',
                            'default' => [],
                        ],
                    ],
                ],
            ]
        );
        
        // Get backups list
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/backup/list',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_backups'],
                    'permission_callback' => [$this, 'get_items_permissions_check'],
                ],
            ]
        );
        
        // Restore backup
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/backup/restore',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'restore_backup'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args' => [
                        'backup_file' => [
                            'required' => false,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                        'filename' => [
                            'required' => false,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                    ],
                ],
            ]
        );
        
        // Delete backup
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/backup/delete',
            [
                [
                    'methods'             => WP_REST_Server::DELETABLE,
                    'callback'            => [$this, 'delete_backup'],
                    'permission_callback' => [$this, 'create_item_permissions_check'],
                    'args' => [
                        'backup_file' => [
                            'required' => false,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                        'filename' => [
                            'required' => false,
                            'type' => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ],
                    ],
                ],
            ]
        );
    }
    
    /**
     * Get module instance
     *
     * @return \ProRank\SEO\Modules\Performance\DatabaseOptimizationModule|null
     */
    private function get_module() {
        if (null === $this->module) {
            $plugin = \ProRank\SEO\Plugin::get_instance();
            if (!$plugin || !$plugin->is_initialized()) {
                return null;
            }
            
            $module_manager = $plugin->modules();
            if (!$module_manager) {
                return null;
            }
            
            $module = $module_manager->get_module('database_optimization');
            if ($module && $module instanceof \ProRank\SEO\Modules\Performance\DatabaseOptimizationModule) {
                $this->module = $module;
            }
        }
        
        return $this->module;
    }
    
    /**
     * Check permissions for reading
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error
     */
    public function get_items_permissions_check($request) {
        if (!current_user_can('manage_options')) {
            return new WP_Error(
                'rest_forbidden',
                __('Sorry, you are not allowed to view database optimization data.', 'prorank-seo'),
                ['status' => 403]
            );
        }
        
        return true;
    }
    
    /**
     * Check permissions for creating
     *
     * @param WP_REST_Request $request Request object
     * @return bool|WP_Error
     */
    public function create_item_permissions_check($request) {
        if (!current_user_can('manage_options')) {
            return new WP_Error(
                'rest_forbidden',
                __('Sorry, you are not allowed to manage database optimization.', 'prorank-seo'),
                ['status' => 403]
            );
        }
        
        return true;
    }
    
    /**
     * Analyze database
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function analyze_database(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        if (!$module->is_enabled()) {
            return new WP_Error(
                'module_disabled',
                __('Database optimization module is disabled.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $analysis = $module->analyze_database();
        
        return rest_ensure_response($analysis);
    }
    
    /**
     * Analyze autoload options
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function analyze_autoload(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        if (!$module->is_enabled()) {
            return new WP_Error(
                'module_disabled',
                __('Database optimization module is disabled.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $analysis = $module->analyze_autoload();
        
        return rest_ensure_response($analysis);
    }
    
    /**
     * Get database statistics
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_database_stats(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        if (!$module->is_enabled()) {
            return new WP_Error(
                'module_disabled',
                __('Database optimization module is disabled.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $stats = $module->get_database_stats();
        
        return rest_ensure_response($stats);
    }
    
    /**
     * Get table overhead information
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_table_overhead(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        if (!$module->is_enabled()) {
            return new WP_Error(
                'module_disabled',
                __('Database optimization module is disabled.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        $overhead = $module->get_table_overhead();
        
        return rest_ensure_response($overhead);
    }
    
    /**
     * Cleanup database
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function cleanup_database(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        if (!$module->is_enabled()) {
            return new WP_Error(
                'module_disabled',
                __('Database optimization module is disabled.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        // Sanitize input
        $type = sanitize_text_field($request->get_param('type'));
        
        // Cleanup via AJAX handler
        $_REQUEST['nonce'] = wp_create_nonce('prorank_database_optimization');
        $_POST['type'] = $type;
        $module->handle_cleanup_database();
        
        // WordPress will handle the response
        die();
    }
    
    /**
     * Get database overview
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_database_overview(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        $overview = $module->get_database_overview();
        
        return rest_ensure_response($overview);
    }
    
    /**
     * Get tables information
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_tables_info(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        $tables = $module->get_all_tables_info();
        
        return rest_ensure_response($tables);
    }
    
    /**
     * Perform table operation
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function table_operation(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Sanitize inputs (defense in depth with schema sanitization)
        $action = sanitize_key($request->get_param('action'));
        $table_name = sanitize_text_field($request->get_param('table_name') ?: $request->get_param('table'));
        
        $result = [];

        if (empty($table_name)) {
            return new WP_Error(
                'invalid_table',
                __('Table name is required.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        switch ($action) {
            case 'repair':
                $result = $module->repair_table($table_name);
                break;
            case 'empty':
                $result = $module->empty_table($table_name);
                break;
            case 'optimize':
                $result = $module->optimize_table($table_name);
                break;
            case 'convert':
                $result = $module->convert_table_to_innodb($table_name);
                break;
        }
        
        if (!$result['success']) {
            return new WP_Error(
                'operation_failed',
                $result['message'],
                ['status' => 400]
            );
        }
        
        return rest_ensure_response($result);
    }
    
    /**
     * Get options
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_options(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Sanitize all inputs
        $args = [
            'search' => sanitize_text_field($request->get_param('search')),
            'autoload' => sanitize_key($request->get_param('autoload')),
            'limit' => absint($request->get_param('limit')),
            'offset' => absint($request->get_param('offset')),
        ];
        
        $options = $module->get_all_options($args);
        
        return rest_ensure_response($options);
    }
    
    /**
     * Update option autoload
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function update_option_autoload(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Sanitize inputs (defense in depth with schema sanitization)
        $option_name = sanitize_text_field($request->get_param('option_name'));
        $autoload = sanitize_key($request->get_param('autoload'));
        
        $success = $module->update_option_autoload($option_name, $autoload);
        
        if (!$success) {
            return new WP_Error(
                'update_failed',
                __('Failed to update option autoload status.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        return rest_ensure_response(['success' => true]);
    }
    
    /**
     * Get cron jobs
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_cron_jobs(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        $jobs = $module->get_all_cron_jobs();
        
        return rest_ensure_response($jobs);
    }
    
    /**
     * Delete cron job
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_cron_job(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Sanitize inputs (defense in depth with schema sanitization)
        $hook = sanitize_text_field($request->get_param('hook'));
        $timestamp = absint($request->get_param('timestamp'));
        
        $success = $module->delete_cron_job($hook, $timestamp);
        
        if (!$success) {
            return new WP_Error(
                'delete_failed',
                __('Failed to delete cron job.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        return rest_ensure_response(['success' => true]);
    }
    
    /**
     * Get orphaned items preview
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_orphaned_preview(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Sanitize input
        $type = sanitize_key($request->get_param('type'));
        
        $items = $module->get_orphaned_items_preview($type);
        
        return rest_ensure_response($items);
    }
    
    /**
     * Create database backup
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function create_backup(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // Sanitize input array
        $tables = array_map('sanitize_text_field', (array) $request->get_param('tables'));
        $result = $module->create_backup($tables);
        
        if (!$result['success']) {
            return new WP_Error(
                'backup_failed',
                $result['message'],
                ['status' => 400]
            );
        }
        
        return rest_ensure_response($result);
    }
    
    /**
     * Get backups list
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_backups(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }
        
        $backups = $module->get_backups();
        
        // Transform backup data for frontend (no public URLs for security)
        $formatted_backups = array_map(function($backup) {
            $size = isset($backup['size']) ? (int) $backup['size'] : 0;
            return [
                'filename' => $backup['file'],
                'created' => $backup['created'] ?? ($backup['date'] ?? ''),
                'date' => $backup['date'] ?? ($backup['created'] ?? ''),
                'size' => $size,
                'size_formatted' => $backup['size_formatted'] ?? size_format($size, 2),
                'tables' => $backup['tables'] ?? 0,
                'storage' => $backup['storage'] ?? 'secure',
                'hash' => $backup['hash'] ?? '',
            ];
        }, $backups);
        
        return rest_ensure_response($formatted_backups);
    }
    
    /**
     * Restore database backup
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function restore_backup(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // CRITICAL: Sanitize file path to prevent path traversal
        $backup_param = $request->get_param('backup_file');
        if (empty($backup_param)) {
            $backup_param = $request->get_param('filename'); // Backward compatibility
        }

        $backup_file = sanitize_file_name($backup_param);
        if (empty($backup_file)) {
            return new WP_Error(
                'invalid_backup',
                __('Backup filename is required.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        $result = $module->restore_backup($backup_file);
        
        if (!$result['success']) {
            return new WP_Error(
                'restore_failed',
                $result['message'],
                ['status' => 400]
            );
        }
        
        return rest_ensure_response($result);
    }
    
    /**
     * Delete backup
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function delete_backup(WP_REST_Request $request) {
        $module = $this->get_module();
        if (!$module) {
            return new WP_Error(
                'module_not_found',
                __('Database optimization module not found.', 'prorank-seo'),
                ['status' => 404]
            );
        }

        // CRITICAL: Sanitize file path to prevent path traversal
        $backup_param = $request->get_param('backup_file');
        if (empty($backup_param)) {
            $backup_param = $request->get_param('filename'); // Backward compatibility
        }

        $backup_file = sanitize_file_name($backup_param);
        if (empty($backup_file)) {
            return new WP_Error(
                'invalid_backup',
                __('Backup filename is required.', 'prorank-seo'),
                ['status' => 400]
            );
        }

        $success = $module->delete_backup($backup_file);
        
        if (!$success) {
            return new WP_Error(
                'delete_failed',
                __('Failed to delete backup.', 'prorank-seo'),
                ['status' => 400]
            );
        }
        
        return rest_ensure_response(['success' => true]);
    }
}
