<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * Click Analytics REST Endpoint
 * 
 * Provides real click tracking data for internal links
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;

// Register the endpoint
add_action('rest_api_init', function() {
    register_rest_route('prorank-seo/v1', '/analytics/link-clicks', [
        [
            'methods' => WP_REST_Server::READABLE,
            'callback' => __NAMESPACE__ . '\get_link_click_analytics',
            'permission_callback' => function() {
                return current_user_can('edit_posts');
            },
            'args' => [
                'period' => [
                    'default' => '30',
                    'sanitize_callback' => 'absint',
                ],
            ],
        ]
    ]);
});

/**
 * Get link click analytics data
 */
function get_link_click_analytics(WP_REST_Request $request): WP_REST_Response {
    global $wpdb;
    
    $period = (int) $request->get_param('period');
    $clicks_table = $wpdb->prefix . 'prorank_link_clicks';
    
    // Initialize response data
    $tracking_enabled = function_exists( 'prorank_is_click_tracking_enabled' )
        ? prorank_is_click_tracking_enabled()
        : true;

    $data = [
        'top_clicked' => [],
        'recent_clicks' => [],
        'click_tracking_enabled' => $tracking_enabled,
        'total_clicks_30d' => 0,
        'total_clicks_7d' => 0,
        'total_clicks_period' => 0,
    ];
    
    // Check if table exists
    $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $clicks_table)) === $clicks_table;
    
    if (!$table_exists) {
        // Return empty data if table doesn't exist yet
        return rest_ensure_response([
            'success' => true,
            'data' => $data,
            'message' => 'Click tracking table not initialized yet'
        ]);
    }
    
    // Get total clicks for the period
    $data['total_clicks_period'] = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM %i 
        WHERE clicked_at > DATE_SUB(NOW(), INTERVAL %d DAY)",
        $clicks_table,
        $period
    )) ?: 0;
    
    // Get 30-day total
    $data['total_clicks_30d'] = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM %i 
        WHERE clicked_at > DATE_SUB(NOW(), INTERVAL %d DAY)",
        $clicks_table,
        30
    )) ?: 0;
    
    // Get 7-day total
    $data['total_clicks_7d'] = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM %i 
        WHERE clicked_at > DATE_SUB(NOW(), INTERVAL %d DAY)",
        $clicks_table,
        7
    )) ?: 0;
    
    // Get top clicked links
    $top_clicked = $wpdb->get_results($wpdb->prepare(
        "SELECT 
            source_post_id,
            target_post_id,
            target_url,
            anchor_text,
            COUNT(*) as click_count
        FROM %i
        WHERE clicked_at > DATE_SUB(NOW(), INTERVAL %d DAY)
        GROUP BY source_post_id, target_post_id, anchor_text
        ORDER BY click_count DESC
        LIMIT 10",
        $clicks_table,
        $period
    ));
    
    // Format top clicked data
    foreach ($top_clicked as $link) {
        $source_post = get_post($link->source_post_id);
        $target_post = get_post($link->target_post_id);
        
        $data['top_clicked'][] = [
            'source_title' => $source_post ? $source_post->post_title : 'Unknown',
            'source_url' => $source_post ? get_permalink($source_post) : '#',
            'target_title' => $target_post ? $target_post->post_title : 'External Link',
            'target_url' => $target_post ? get_permalink($target_post) : $link->target_url,
            'anchor_text' => $link->anchor_text ?: 'Click here',
            'clicks' => (int) $link->click_count,
        ];
    }
    
    // Get recent clicks
    $recent_clicks = $wpdb->get_results($wpdb->prepare(
        "SELECT 
            source_post_id,
            target_post_id,
            target_url,
            anchor_text,
            user_id,
            clicked_at
        FROM %i
        ORDER BY clicked_at DESC
        LIMIT %d",
        $clicks_table,
        20
    ));
    
    // Format recent clicks
    foreach ($recent_clicks as $click) {
        $source_post = get_post($click->source_post_id);
        $target_post = get_post($click->target_post_id);
        $user = $click->user_id ? get_userdata($click->user_id) : null;
        
        $data['recent_clicks'][] = [
            'time' => $click->clicked_at,
            'source' => $source_post ? $source_post->post_title : 'Unknown',
            'target' => $target_post ? $target_post->post_title : 'External',
            'user' => $user ? $user->display_name : __('Guest', 'prorank-seo'),
        ];
    }
    
    // If no real data yet, provide helpful message
    if (empty($data['top_clicked']) && empty($data['recent_clicks'])) {
        $data['message'] = 'No click data collected yet. Links will be tracked as users click them.';
        
        // Check if tracking is enabled
        if (!$data['click_tracking_enabled']) {
            $data['message'] = 'Click tracking is disabled. Enable it in settings to start collecting data.';
        }
    }
    
    return rest_ensure_response([
        'success' => true,
        'data' => $data,
    ]);
}
