<?php
/**
 * Cache REST API Controller
 *
 * Handles cache-related endpoints
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;
use ProRank\SEO\Plugin;

/**
 * Cache controller class
 */
class CacheController extends BaseController {
    
    /**
     * Rest base
     *
     * @var string
     */
    protected $rest_base = 'cache';
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        // Clear cache endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/clear',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'clear_cache'],
                    'permission_callback' => [$this, 'check_permission'],
                    'args'                => [
                        'type' => [
                            'required'          => false,
                            'type'              => 'string',
                            'default'           => 'all',
                            'enum'              => ['all', 'page', 'post', 'edge'],
                            'sanitize_callback' => 'sanitize_key',
                        ],
                        'id' => [
                            'required'          => false,
                            'type'              => 'integer',
                            'sanitize_callback' => 'absint',
                        ],
                    ],
                ],
            ]
        );
        
        // Preload cache endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/preload',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'preload_cache'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Get cache status endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/status',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_cache_status'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Get cache statistics endpoint (NEW for 2025)
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/stats',
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_cache_stats'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Test speed endpoint (NEW for 2025)
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/test-speed',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'test_speed'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Cancel preload endpoint
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/preload/cancel',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'cancel_preload'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
    }
    
    /**
     * Clear cache
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function clear_cache(WP_REST_Request $request) {
        // Sanitize inputs (defense in depth with schema sanitization)
        $type = sanitize_key($request->get_param('type'));
        $id = absint($request->get_param('id'));

        try {
            $plugin = Plugin::get_instance();
            $module_manager = $plugin->modules();
            
            // Check if cache module is available
            $cache_module = $module_manager->get_module('cache');
            if (!$cache_module) {
                $this->clear_plugin_caches();
                $message = __('ProRank caches cleared successfully', 'prorank-seo');

                // Trigger action after cache clear
                do_action('prorank_seo_cache_cleared', $type, $id);

                return rest_ensure_response([
                    'success' => true,
                    'message' => $message,
                    'type' => $type,
                    'id' => $id,
                ]);
            }
            
            // Clear cache based on type
            switch ($type) {
                case 'post':
                case 'page':
                    if (!$id) {
                        return $this->get_error_response(
                            __('Post ID required for single post cache clear', 'prorank-seo'),
                            400
                        );
                    }
                    $cache_module->clear_post_cache($id);
                    $message = sprintf(
                        /* translators: %d: post ID */
                        __('Cache cleared for post %d', 'prorank-seo'),
                        $id
                    );
                    break;
                    
                case 'all':
                default:
                    $cache_module->clear_all_cache();
                    $message = __('All cache cleared successfully', 'prorank-seo');
                    break;
            }
            
            // Trigger action after cache clear
            do_action('prorank_seo_cache_cleared', $type, $id);
            
            return rest_ensure_response([
                'success' => true,
                'message' => $message,
                'type' => $type,
                'id' => $id,
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to clear cache: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }

    /**
     * Clear plugin-level caches when cache module is unavailable.
     *
     * @return void
     */
    private function clear_plugin_caches(): void {
        // Clear CSS cache files.
        do_action('prorank_css_cache_cleared');

        // Clear known cache groups when supported by object cache.
        if (function_exists('wp_cache_flush_group')) {
            wp_cache_flush_group('prorank_redirects');
            wp_cache_flush_group('prorank_seo');
        }

        $this->clear_prorank_transients();
    }

    /**
     * Delete ProRank transients from the options table.
     *
     * @return void
     */
    private function clear_prorank_transients(): void {
        global $wpdb;

        $transient_like = $wpdb->esc_like('_transient_prorank') . '%';
        $timeout_like = $wpdb->esc_like('_transient_timeout_prorank') . '%';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s OR option_name LIKE %s",
                $transient_like,
                $timeout_like
            )
        );
    }
    
    /**
     * Start cache preloading
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function preload_cache(WP_REST_Request $request) {
        try {
            $plugin = Plugin::get_instance();
            
            // Get cache preload module
            $module_manager = $plugin->modules();
            $preload_module = $module_manager->get_module('cache_preload');
            
            if (!$preload_module || !$preload_module->is_enabled()) {
                return $this->get_error_response(
                    __('Cache preloading module is not enabled', 'prorank-seo'),
                    400
                );
            }
            
            // Start preload
            $success = $preload_module->start_preload(true);
            
            if (!$success) {
                return $this->get_error_response(
                    __('Failed to start cache preloading', 'prorank-seo'),
                    500
                );
            }
            
            return rest_ensure_response([
                'success' => true,
                'message' => __('Cache preloading started', 'prorank-seo'),
                'status' => $preload_module->get_status(),
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to start preloading: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }
    
    /**
     * Get cache status
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_cache_status(WP_REST_Request $request) {
        try {
            $cache_dir = prorank_page_cache_dir();
            $status = [
                'enabled' => false,
                'size' => 0,
                'files' => 0,
                'last_cleared' => null,
                'preload_status' => null,
            ];
            
            // Check if cache directory exists
            if (is_dir($cache_dir)) {
                $status['enabled'] = true;
                
                // Calculate cache size and file count
                $iterator = new \RecursiveIteratorIterator(
                    new \RecursiveDirectoryIterator($cache_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                    \RecursiveIteratorIterator::LEAVES_ONLY
                );
                
                foreach ($iterator as $file) {
                    if ($file->isFile()) {
                        $status['size'] += $file->getSize();
                        $status['files']++;
                    }
                }
            }
            
            // Get last cleared time from option
            $status['last_cleared'] = get_option('prorank_cache_last_cleared', null);
            
            // Get preload status from module
            $plugin = Plugin::get_instance();
            $module_manager = $plugin->modules();
            $preload_module = $module_manager->get_module('cache_preload');
            
            if ($preload_module && $preload_module->is_enabled()) {
                $status['preload_status'] = $preload_module->get_status();
            }
            
            return rest_ensure_response($status);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to get cache status: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }
    
    /**
     * Cancel cache preloading
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function cancel_preload(WP_REST_Request $request) {
        try {
            // Get cache preload module
            $plugin = Plugin::get_instance();
            $module_manager = $plugin->modules();
            $preload_module = $module_manager->get_module('cache_preload');
            
            if (!$preload_module || !$preload_module->is_enabled()) {
                return $this->get_error_response(
                    __('Cache preloading module is not enabled', 'prorank-seo'),
                    400
                );
            }
            
            // Cancel preload
            $preload_module->cancel_preload();
            
            return rest_ensure_response([
                'success' => true,
                'message' => __('Cache preloading cancelled', 'prorank-seo'),
            ]);
            
        } catch (\Exception $e) {
            return $this->get_error_response(
                sprintf(
                    /* translators: %s: error message */
                    __('Failed to cancel preloading: %s', 'prorank-seo'), esc_html($e->getMessage())),
                500
            );
        }
    }
    
    /**
     * Get cache statistics (NEW for 2025)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function get_cache_stats(WP_REST_Request $request) {
        try {
            $plugin = Plugin::get_instance();
            $module_manager = $plugin->modules();
            
            // Try to get modern cache module first
            $cache_module = $module_manager->get_module('modern_cache');
            if (!$cache_module) {
                // Fall back to regular cache module
                $cache_module = $module_manager->get_module('cache');
            }
            
            if (!$cache_module) {
                // Return default stats if no cache module
                return rest_ensure_response([
                    'hit_rate' => 0,
                    'miss_rate' => 0,
                    'total_cached' => 0,
                    'cache_size' => '0 MB',
                    'bandwidth_saved' => '0 GB',
                    'avg_response_time' => '0ms',
                    'edge_locations' => 0,
                ]);
            }
            
            // Get stats from cache module
            $stats = [];
            if (method_exists($cache_module, 'get_cache_stats')) {
                $stats = $cache_module->get_cache_stats();
            } else {
                // Calculate basic stats if method doesn't exist
                $cache_dir = prorank_page_cache_dir();
                $total_size = 0;
                $file_count = 0;
                
                if (is_dir($cache_dir)) {
                    $iterator = new \RecursiveIteratorIterator(
                        new \RecursiveDirectoryIterator($cache_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                        \RecursiveIteratorIterator::LEAVES_ONLY
                    );
                    
                    foreach ($iterator as $file) {
                        if ($file->isFile()) {
                            $total_size += $file->getSize();
                            if ($file->getExtension() === 'html') {
                                $file_count++;
                            }
                        }
                    }
                }
                
                // Get real stats from local tracking and ProRank.io
                $local_stats = $this->get_local_cache_stats();
                $remote_stats = $this->get_remote_stats();
                
                // Merge local and remote stats for comprehensive metrics
                $stats = [
                    'hit_rate' => $local_stats['hit_rate'] ?? $remote_stats['hit_rate'] ?? 0,
                    'miss_rate' => $local_stats['miss_rate'] ?? $remote_stats['miss_rate'] ?? 0,
                    'total_cached' => $file_count,
                    'cache_size' => $this->format_bytes($total_size),
                    'bandwidth_saved' => $this->calculate_bandwidth_saved($total_size, $local_stats),
                    'avg_response_time' => $local_stats['avg_response_time'] ?? $remote_stats['avg_response_time'] ?? 'N/A',
                    'edge_locations' => $remote_stats['edge_locations'] ?? $this->get_tier_edge_locations(),
                    'cache_efficiency' => $this->calculate_cache_efficiency($local_stats),
                    'compression_ratio' => $this->calculate_compression_ratio($cache_dir),
                    'real_time_status' => $this->get_real_time_status(),
                ];
            }
            
            return rest_ensure_response($stats);
        } catch (\Exception $e) {
            return $this->get_error_response(
                esc_html($e->getMessage()),
                500
            );
        }
    }
    
    /**
     * Test speed (NEW for 2025)
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response|WP_Error
     */
    public function test_speed(WP_REST_Request $request) {
        try {
            $home_url = home_url('/');
            
            // Test without cache (add random param to bypass cache)
            $start_no_cache = microtime(true);
            $response_no_cache = wp_remote_get($home_url . '?nocache=' . time(), [
                'timeout' => 10,
                'sslverify' => true,
            ]);
            $time_no_cache = (microtime(true) - $start_no_cache) * 1000;
            
            // Test with cache (should hit cache if enabled)
            $start_cache = microtime(true);
            $response_cache = wp_remote_get($home_url, [
                'timeout' => 10,
                'sslverify' => true,
            ]);
            $time_cache = (microtime(true) - $start_cache) * 1000;
            
            // Check if cache header is present
            $headers = wp_remote_retrieve_headers($response_cache);
            $cache_hit = isset($headers['x-prorank-cache']) && $headers['x-prorank-cache'] === 'HIT';
            
            // Calculate improvement
            $improvement = 0;
            if ($time_no_cache > 0) {
                $improvement = round((($time_no_cache - $time_cache) / $time_no_cache) * 100, 2);
            }
            
            $result = [
                'cache_enabled' => $cache_hit,
                'time_without_cache' => round($time_no_cache, 2) . 'ms',
                'time_with_cache' => round($time_cache, 2) . 'ms',
                'improvement' => $improvement . '%',
                'compression' => $headers['x-prorank-compression'] ?? 'none',
                'recommendations' => [],
            ];
            
            // Add recommendations
            if (!$cache_hit) {
                $result['recommendations'][] = __('Enable page caching for better performance', 'prorank-seo');
            }
            
            if ($improvement < 50 && $cache_hit) {
                $result['recommendations'][] = __('Consider enabling edge caching for better global performance', 'prorank-seo');
            }
            
            if (!isset($headers['content-encoding']) || strpos($headers['content-encoding'], 'br') === false) {
                $result['recommendations'][] = __('Enable Brotli compression for 20-30% better compression', 'prorank-seo');
            }
            
            return rest_ensure_response($result);
        } catch (\Exception $e) {
            return $this->get_error_response(
                esc_html($e->getMessage()),
                500
            );
        }
    }

    /**
     * Format bytes to human readable
     *
     * @param int $bytes Bytes to format
     * @return string
     */
    private function format_bytes(int $bytes): string {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $i = 0;
        
        while ($bytes >= 1024 && $i < count($units) - 1) {
            $bytes /= 1024;
            $i++;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
    
    /**
     * Get remote stats from ProRank.io
     *
     * @return array
     */
    private function get_remote_stats(): array {
        return [];
    }
    
    /**
     * Get local cache statistics
     *
     * @return array
     */
    private function get_local_cache_stats(): array {
        // Get stats from WordPress transients (updated on each cache hit/miss)
        $stats = get_transient('prorank_cache_stats');
        
        if ($stats === false) {
            // Initialize stats if not exists
            $stats = [
                'hits' => 0,
                'misses' => 0,
                'bytes_served' => 0,
                'response_times' => [],
                'last_updated' => time(),
            ];
            set_transient('prorank_cache_stats', $stats, HOUR_IN_SECONDS);
        }
        
        $total = $stats['hits'] + $stats['misses'];
        $hit_rate = $total > 0 ? round(($stats['hits'] / $total) * 100, 2) : 0;
        
        // Calculate average response time
        $avg_response = 0;
        if (!empty($stats['response_times'])) {
            $avg_response = round(array_sum($stats['response_times']) / count($stats['response_times']), 2);
        }
        
        return [
            'hit_rate' => $hit_rate,
            'miss_rate' => 100 - $hit_rate,
            'total_hits' => $stats['hits'],
            'total_misses' => $stats['misses'],
            'bytes_served' => $stats['bytes_served'],
            'avg_response_time' => $avg_response > 0 ? $avg_response . 'ms' : null,
        ];
    }
    
    /**
     * Calculate bandwidth saved
     *
     * @param int   $cache_size Total cache size
     * @param array $stats      Local stats
     * @return string
     */
    private function calculate_bandwidth_saved(int $cache_size, array $stats): string {
        // Calculate based on cache hits and average page size
        $hits = $stats['total_hits'] ?? 0;
        $avg_page_size = $cache_size > 0 && $stats['total_hits'] > 0 
            ? $cache_size / max(1, $stats['total_hits']) 
            : 50000; // 50KB average
        
        $bandwidth_saved = $hits * $avg_page_size * 10; // Multiply by 10 for network overhead
        
        return $this->format_bytes($bandwidth_saved);
    }
    
    /**
     * Calculate cache efficiency
     *
     * @param array $stats Local stats
     * @return string
     */
    private function calculate_cache_efficiency(array $stats): string {
        $hit_rate = $stats['hit_rate'] ?? 0;
        
        if ($hit_rate >= 90) {
            return 'Excellent';
        } elseif ($hit_rate >= 70) {
            return 'Good';
        } elseif ($hit_rate >= 50) {
            return 'Fair';
        } else {
            return 'Building';
        }
    }
    
    /**
     * Calculate compression ratio
     *
     * @param string $cache_dir Cache directory
     * @return string
     */
    private function calculate_compression_ratio(string $cache_dir): string {
        $total_original = 0;
        $total_compressed = 0;
        
        if (is_dir($cache_dir)) {
            $iterator = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator($cache_dir, \RecursiveDirectoryIterator::SKIP_DOTS),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );
            
            foreach ($iterator as $file) {
                if ($file->isFile()) {
                    $ext = $file->getExtension();
                    if ($ext === 'html') {
                        $total_original += $file->getSize();
                    } elseif ($ext === 'gz' || $ext === 'br') {
                        $total_compressed += $file->getSize();
                    }
                }
            }
        }
        
        if ($total_original > 0 && $total_compressed > 0) {
            $ratio = round((1 - ($total_compressed / $total_original)) * 100, 2);
            return $ratio . '%';
        }
        
        return 'N/A';
    }
    
    /**
     * Get real-time cache status
     *
     * @return array
     */
    private function get_real_time_status(): array {
        return [
            'active' => get_option('prorank_cache_enabled', false),
            'preloading' => get_option('prorank_preload_running', false),
            'last_cleared' => get_option('prorank_cache_last_cleared', null),
            'optimization_score' => $this->calculate_optimization_score(),
        ];
    }
    
    /**
     * Calculate optimization score
     *
     * @return int
     */
    private function calculate_optimization_score(): int {
        $score = 0;
        
        // Check various optimization settings
        if (get_option('prorank_cache_enabled')) $score += 20;
        if (get_option('prorank_gzip_enabled')) $score += 15;
        if (get_option('prorank_brotli_enabled')) $score += 20;
        if (get_option('prorank_preload_enabled')) $score += 15;
        if (get_option('prorank_edge_enabled')) $score += 20;
        if (get_option('prorank_minify_enabled')) $score += 10;
        
        return min(100, $score);
    }
    
    /**
     * Get tier edge locations
     *
     * @return int
     */
    private function get_tier_edge_locations(): int {
        return 0;
    }
}
