<?php
/**
 * Audit Settings REST Controller
 *
 * Handles audit settings persistence
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Controller;
use ProRank\SEO\Core\AuditSettings;

class AuditSettingsController extends WP_REST_Controller {
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->namespace = 'prorank-seo/v1';
        $this->rest_base = 'audit-settings';
    }
    
    /**
     * Register routes
     */
    public function register_routes() {
        // Get settings
        register_rest_route($this->namespace, '/' . $this->rest_base, [
            'methods' => \WP_REST_Server::READABLE,
            'callback' => [$this, 'get_settings'],
            'permission_callback' => [$this, 'check_permission'],
        ]);
        
        // Save settings
        register_rest_route($this->namespace, '/' . $this->rest_base, [
            'methods' => \WP_REST_Server::CREATABLE,
            'callback' => [$this, 'save_settings'],
            'permission_callback' => [$this, 'check_permission'],
        ]);
        
        // Apply preset
        register_rest_route($this->namespace, '/' . $this->rest_base . '/preset', [
            'methods' => \WP_REST_Server::CREATABLE,
            'callback' => [$this, 'apply_preset'],
            'permission_callback' => [$this, 'check_permission'],
            'args' => [
                'preset' => [
                    'required' => true,
                    'type' => 'string',
                    'enum' => ['fast', 'balanced', 'deep', 'ecommerce', 'blog'],
                ],
            ],
        ]);
        
        // Get presets
        register_rest_route($this->namespace, '/' . $this->rest_base . '/presets', [
            'methods' => \WP_REST_Server::READABLE,
            'callback' => [$this, 'get_presets'],
            'permission_callback' => [$this, 'check_permission'],
        ]);
    }
    
    /**
     * Check permission
     */
    public function check_permission() {
        return current_user_can('manage_options');
    }
    
    /**
     * Get settings
     */
    public function get_settings($request) {
        $settings = AuditSettings::get_settings();
        
        return rest_ensure_response([
            'success' => true,
            'settings' => $settings,
        ]);
    }
    
    /**
     * Save settings
     */
    public function save_settings($request) {
        $settings = $request->get_json_params();
        
        if (empty($settings)) {
            return new \WP_Error('invalid_settings', __('Invalid settings data', 'prorank-seo'));
        }
        
        $saved = AuditSettings::save_settings($settings);
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Settings saved successfully', 'prorank-seo'),
            'settings' => $saved,
        ]);
    }
    
    /**
     * Apply preset
     */
    public function apply_preset($request) {
        $preset = $request->get_param('preset');
        
        $result = AuditSettings::apply_preset($preset);
        
        if ($result === false) {
            return new \WP_Error('invalid_preset', __('Invalid preset', 'prorank-seo'));
        }
        
        return rest_ensure_response([
            'success' => true,
            'message' => __('Preset applied successfully', 'prorank-seo'),
            'settings' => $result,
        ]);
    }
    
    /**
     * Get presets
     */
    public function get_presets($request) {
        $presets = AuditSettings::get_presets();
        
        return rest_ensure_response([
            'success' => true,
            'presets' => $presets,
        ]);
    }
}