<?php
/**
 * Analytics Click Controller
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Request;
use WP_REST_Response;

/**
 * AnalyticsClickController class
 */
class AnalyticsClickController extends BaseController {
    
    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        $this->rest_base = 'analytics/link-clicks';
    }
    
    /**
     * Register routes
     *
     * @return void
     */
    public function register_routes(): void {
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_link_clicks'],
                    'permission_callback' => [$this, 'check_admin_permission'],
                    'args'                => [
                        'period' => [
                            'description' => 'Time period in days',
                            'type'        => 'integer',
                            'default'     => 30,
                        ],
                    ],
                ],
            ]
        );
    }
    
    /**
     * Get link click analytics
     *
     * @param WP_REST_Request $request Request object
     * @return WP_REST_Response
     */
    public function get_link_clicks(WP_REST_Request $request) {
        $period = (int) $request->get_param('period');
        
        // Mock data for now
        $mock_data = [
            'top_clicked' => [
                [
                    'source_url' => home_url('/seo-guide/'),
                    'source_title' => 'Complete SEO Guide',
                    'target_url' => home_url('/keyword-research/'),
                    'target_title' => 'Keyword Research Tutorial',
                    'anchor_text' => 'keyword research',
                    'click_count' => 234,
                ],
                [
                    'source_url' => home_url('/blog/'),
                    'source_title' => 'Blog Homepage',
                    'target_url' => home_url('/seo-guide/'),
                    'target_title' => 'Complete SEO Guide',
                    'anchor_text' => 'SEO guide',
                    'click_count' => 189,
                ],
                [
                    'source_url' => home_url('/about/'),
                    'source_title' => 'About Us',
                    'target_url' => home_url('/services/'),
                    'target_title' => 'Our Services',
                    'anchor_text' => 'services',
                    'click_count' => 156,
                ],
            ],
            'recent_clicks' => [
                [
                    'clicked_at' => current_time('mysql'),
                    'source_url' => home_url('/blog/'),
                    'source_title' => 'Blog Homepage',
                    'target_url' => home_url('/seo-guide/'),
                    'target_title' => 'Complete SEO Guide',
                    'user_name' => 'John Doe',
                ],
                [
                    'clicked_at' => gmdate('Y-m-d H:i:s', strtotime('-1 hour')),
                    'source_url' => home_url('/seo-guide/'),
                    'source_title' => 'Complete SEO Guide',
                    'target_url' => home_url('/keyword-research/'),
                    'target_title' => 'Keyword Research Tutorial',
                    'user_name' => null,
                ],
            ],
            'click_tracking_enabled' => function_exists( 'prorank_is_click_tracking_enabled' )
                ? prorank_is_click_tracking_enabled()
                : false,
            'total_clicks_30d' => 1234,
            'total_clicks_7d' => 342,
        ];
        
        return rest_ensure_response([
            'success' => true,
            'data' => $mock_data,
        ]);
    }
}
