<?php
/**
 * Advanced Settings REST API Controller
 *
 * Handles advanced settings for power users including debug, security, and WordPress features
 *
 * @package ProRank\SEO\Core\RestApi
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\RestApi;

defined( 'ABSPATH' ) || exit;

use WP_REST_Server;
use WP_REST_Request;
use WP_REST_Response;
use WP_Error;

/**
 * Advanced Settings controller class
 */
class AdvancedSettingsController extends BaseController {
    
    /**
     * Rest base
     */
    protected $rest_base = 'advanced-settings';
    
    /**
     * Register routes
     */
    public function register_routes(): void {
        // Get/Save advanced settings
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base,
            [
                [
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => [$this, 'get_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'save_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
        
        // Apply settings immediately
        register_rest_route(
            $this->namespace,
            '/' . $this->rest_base . '/apply',
            [
                [
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => [$this, 'apply_settings'],
                    'permission_callback' => [$this, 'check_permission'],
                ],
            ]
        );
    }
    
    /**
     * Get advanced settings
     */
    public function get_settings(WP_REST_Request $request): WP_REST_Response {
        $defaults = $this->get_default_settings();
        $settings = get_option('prorank_seo_advanced_settings', $defaults);

        if (!is_array($settings)) {
            $settings = $defaults;
        }

        $settings = array_merge($defaults, $settings);
        $settings = $this->sanitize_settings($settings);
        
        return new WP_REST_Response([
            'success' => true,
            'data'    => $settings,
        ]);
    }
    
    /**
     * Save advanced settings
     */
    public function save_settings(WP_REST_Request $request): WP_REST_Response {
        $settings = $request->get_param('advanced');
        
        if (!$settings) {
            return new WP_REST_Response([
                'success' => false,
                'message' => __('No settings provided', 'prorank-seo'),
            ], 400);
        }
        
        // Validate and sanitize settings
        $sanitized = $this->sanitize_settings($settings);
        
        // Save to database
        update_option('prorank_seo_advanced_settings', $sanitized);
        
        // Apply settings immediately
        $this->apply_settings_internal($sanitized);
        
        return new WP_REST_Response([
            'success' => true,
            'message' => __('Advanced settings saved successfully', 'prorank-seo'),
            'data'    => $sanitized,
        ]);
    }
    
    /**
     * Apply settings immediately
     */
    public function apply_settings(WP_REST_Request $request): WP_REST_Response {
        $settings = get_option('prorank_seo_advanced_settings', $this->get_default_settings());
        $this->apply_settings_internal($settings);
        
        return new WP_REST_Response([
            'success' => true,
            'message' => __('Settings applied successfully', 'prorank-seo'),
        ]);
    }
    
    /**
     * Apply settings internally
     *
     * Note: Some settings (like WP_DEBUG constants and DISALLOW_FILE_EDIT)
     * cannot be changed at runtime as they must be defined in wp-config.php
     * before WordPress loads. These settings are stored for reference only.
     */
    private function apply_settings_internal(array $settings): void {
        // Debug & Development - Note: WP_DEBUG constants must be set in wp-config.php
        // We can only control plugin-level debug logging
        update_option('prorank_seo_log_level', $settings['log_level']);
        update_option('prorank_seo_debug_mode', !empty($settings['debug_mode']));

        // Performance settings
        if (!empty($settings['api_timeout'])) {
            update_option('prorank_seo_api_timeout', intval($settings['api_timeout']));
        }

        if (!empty($settings['cache_duration'])) {
            update_option('prorank_seo_cache_duration', intval($settings['cache_duration']));
        }

        // Store a flag that settings have been applied
        update_option('prorank_seo_advanced_settings_applied', true);
    }

    /**
     * Apply security and feature settings on init
     * This should be called early in WordPress initialization
     */
    public static function apply_saved_settings(): void {
        $settings = get_option('prorank_seo_advanced_settings', []);
        if (empty($settings)) {
            return;
        }

        if (!is_array($settings)) {
            return;
        }

        $controller = new self();
        $settings = $controller->sanitize_settings($settings);

        // Security features
        if (!empty($settings['disable_xmlrpc'])) {
            add_filter('xmlrpc_enabled', '__return_false');
            add_filter('xmlrpc_methods', function($methods) {
                return [];
            });
        }

        if (!empty($settings['disable_rest_api_public'])) {
            add_filter('rest_authentication_errors', function($result) {
                if (!is_user_logged_in()) {
                    return new \WP_Error(
                        'rest_forbidden',
                        __('REST API restricted to authenticated users.', 'prorank-seo'),
                        ['status' => 401]
                    );
                }
                return $result;
            });
        }

        if (!empty($settings['remove_version_strings'])) {
            // Remove version from scripts and styles
            add_filter('style_loader_src', [__CLASS__, 'static_remove_version_from_url'], 9999);
            add_filter('script_loader_src', [__CLASS__, 'static_remove_version_from_url'], 9999);

            // Remove WordPress version
            remove_action('wp_head', 'wp_generator');
            add_filter('the_generator', '__return_empty_string');
        }

        if (!empty($settings['disable_file_editing'])) {
            add_filter('user_has_cap', function($allcaps) {
                $allcaps['edit_plugins'] = false;
                $allcaps['edit_themes'] = false;
                $allcaps['edit_files'] = false;
                return $allcaps;
            });

            add_action('admin_menu', function() {
                remove_submenu_page('themes.php', 'theme-editor.php');
                remove_submenu_page('plugins.php', 'plugin-editor.php');
            }, 999);
        }

        // WordPress features
        if (!empty($settings['disable_emojis'])) {
            remove_action('wp_head', 'print_emoji_detection_script', 7);
            remove_action('admin_print_scripts', 'print_emoji_detection_script');
            remove_action('wp_print_styles', 'print_emoji_styles');
            remove_action('admin_print_styles', 'print_emoji_styles');
            remove_filter('the_content_feed', 'wp_staticize_emoji');
            remove_filter('comment_text_rss', 'wp_staticize_emoji');
            remove_filter('wp_mail', 'wp_staticize_emoji_for_email');

            add_filter('tiny_mce_plugins', function($plugins) {
                if (is_array($plugins)) {
                    return array_diff($plugins, ['wpemoji']);
                }
                return [];
            });
        }

        if (!empty($settings['disable_gutenberg'])) {
            add_filter('use_block_editor_for_post', '__return_false', 10);
            add_filter('use_block_editor_for_post_type', '__return_false', 10);

            // Remove Gutenberg CSS
            add_action('wp_print_styles', function() {
                wp_dequeue_style('wp-block-library');
                wp_dequeue_style('wp-block-library-theme');
                wp_dequeue_style('wc-block-style');
                wp_dequeue_style('global-styles');
            }, 100);
        }

        // API timeout
        if (!empty($settings['api_timeout'])) {
            add_filter('http_request_timeout', function() use ($settings) {
                return intval($settings['api_timeout']);
            });
        }

        if (!empty($settings['cache_duration'])) {
            add_filter('prorank_seo_breadcrumbs_cache_duration', function() use ($settings) {
                return max(60, intval($settings['cache_duration']));
            });
        }
    }

    /**
     * Static version remover for use with apply_saved_settings
     */
    public static function static_remove_version_from_url($src) {
        if (!is_string($src) || $src === '') {
            return $src;
        }

        if (false !== strpos($src, '?ver=') || false !== strpos($src, '&ver=')) {
            $src = remove_query_arg('ver', $src);
        }

        return $src;
    }
    
    /**
     * Disable WordPress emojis
     */
    private function disable_emojis(): void {
        remove_action('wp_head', 'print_emoji_detection_script', 7);
        remove_action('admin_print_scripts', 'print_emoji_detection_script');
        remove_action('wp_print_styles', 'print_emoji_styles');
        remove_action('admin_print_styles', 'print_emoji_styles');
        remove_filter('the_content_feed', 'wp_staticize_emoji');
        remove_filter('comment_text_rss', 'wp_staticize_emoji');
        remove_filter('wp_mail', 'wp_staticize_emoji_for_email');
        
        add_filter('tiny_mce_plugins', function($plugins) {
            if (is_array($plugins)) {
                return array_diff($plugins, ['wpemoji']);
            }
            return [];
        });
        
        add_filter('wp_resource_hints', function($urls, $relation_type) {
            if ('dns-prefetch' === $relation_type) {
                // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Core filter name.
                $emoji_svg_url = apply_filters('emoji_svg_url', 'https://s.w.org/images/core/emoji/');
                $urls = array_diff($urls, [$emoji_svg_url]);
            }
            return $urls;
        }, 10, 2);
    }
    
    /**
     * Remove version from URL
     */
    public function remove_version_from_url($src) {
        if (strpos($src, '?ver=') || strpos($src, '&ver=')) {
            $src = remove_query_arg('ver', $src);
        }
        return $src;
    }
    
    /**
     * Sanitize settings
     */
    private function sanitize_settings(array $settings): array {
        $sanitized = [];
        
        // Boolean settings
        $boolean_keys = [
            'debug_mode',
            'disable_emojis',
            'disable_gutenberg',
            'disable_xmlrpc',
            'disable_rest_api_public',
            'remove_version_strings',
            'disable_file_editing',
        ];
        
        foreach ($boolean_keys as $key) {
            $sanitized[$key] = !empty($settings[$key]);
        }
        
        // String settings
        $sanitized['log_level'] = in_array($settings['log_level'] ?? 'error', ['error', 'warning', 'info', 'debug']) 
            ? $settings['log_level'] 
            : 'error';
        
        // Numeric settings
        $sanitized['api_timeout'] = min(120, max(5, absint($settings['api_timeout'] ?? 30)));
        $sanitized['cache_duration'] = min(86400, max(60, absint($settings['cache_duration'] ?? 3600)));
        
        return $sanitized;
    }
    
    /**
     * Get default settings
     */
    private function get_default_settings(): array {
        return [
            'debug_mode' => false,
            'disable_emojis' => false,
            'disable_gutenberg' => false,
            'disable_xmlrpc' => true,
            'disable_rest_api_public' => false,
            'remove_version_strings' => true,
            'disable_file_editing' => true,
            'api_timeout' => 30,
            'cache_duration' => 3600,
            'log_level' => 'error',
        ];
    }
}
