<?php
/**
 * CSS Minifier Service
 *
 * Handles CSS minification using Matthias Mullie's Minify library
 *
 * @package ProRank\SEO\Core\Optimization\CSS
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Optimization\CSS;

defined( 'ABSPATH' ) || exit;

use MatthiasMullie\Minify\CSS as CSSMinify;
use Exception;

/**
 * CssMinifierService class
 */
class CssMinifierService {
    
    /**
     * Cache directory for minified files
     *
     * @var string
     */
    private string $cache_dir;
    
    /**
     * Cache URL for minified files
     *
     * @var string
     */
    private string $cache_url;
    
    /**
     * Options for minification
     *
     * @var array
     */
    private array $options;
    
    /**
     * Constructor
     *
     * @param array $options Minification options
     */
    public function __construct(array $options = []) {
        $this->options = wp_parse_args($options, [
            'gzip' => true,
            'preserve_comments' => false,
            'remove_important_comments' => false,
        ]);

        // Keep cache path aligned with the public rewrite (/prorank-cache/css)
        $upload_dir = wp_upload_dir();
        $this->cache_dir = trailingslashit($upload_dir['basedir']) . 'prorank-cache/css/';
        $this->cache_url = trailingslashit($upload_dir['baseurl']) . 'prorank-cache/css/';
        
        $this->ensure_cache_directory();
    }
    
    /**
     * Minify CSS content
     *
     * @param string $css     CSS content to minify
     * @param string $source  Source file path (for path rewriting)
     * @return string Minified CSS
     * @throws Exception If minification fails
     */
    public function minify(string $css, string $source = ''): string {
        try {
            // IMPORTANT: Add font-display:swap BEFORE minification
            // This ensures all @font-face rules have optimal font loading behavior
            $css = $this->addFontDisplaySwap($css);

            $this->ensureMinifyLibrary();

            $minifier = new CSSMinify($css);

            // Set options
            if ($this->options['preserve_comments']) {
                // This will preserve important comments (/*! ... */)
                $minifier->setImportExtensions([]);
            }

            $minified = $minifier->minify();
            $is_original_empty = trim($css) === '';
            $is_minified_empty = trim($minified) === '';

            // If minifier stripped everything but the source had content, keep original to avoid blank styles
            if ($is_minified_empty && !$is_original_empty) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    prorank_log('ProRank CSS Minification Warning: Minification resulted in empty content, using original.');
                }
                return $css;
            }

            return $minified;
        } catch (Exception $e) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                prorank_log('ProRank CSS Minification Error: ' . $e->getMessage());
            }
            throw $e;
        }
    }

    /**
     * Ensure the Matthias Mullie minifier classes are loaded.
     *
     * @throws Exception When the library cannot be loaded.
     */
    private function ensureMinifyLibrary(): void {
        if ( class_exists( CSSMinify::class ) ) {
            return;
        }

        $base = dirname( __DIR__, 3 ) . '/lib/MatthiasMullie/Minify/';
        $path_converter_base = dirname( __DIR__, 3 ) . '/lib/MatthiasMullie/PathConverter/';
        $files = [
            'Exception.php',
            'Minify.php',
            'CSS.php',
            'JS.php',
        ];
        $path_converter_files = [
            'ConverterInterface.php',
            'NoConverter.php',
            'Converter.php',
        ];

        foreach ( $path_converter_files as $file ) {
            $path = $path_converter_base . $file;
            if ( file_exists( $path ) ) {
                require_once $path;
            }
        }

        foreach ( $files as $file ) {
            $path = $base . $file;
            if ( file_exists( $path ) ) {
                require_once $path;
            }
        }

        if ( ! class_exists( CSSMinify::class ) ) {
            throw new Exception( 'ProRank CSS Minification Error: MatthiasMullie Minify library not found.' );
        }
    }

    /**
     * Add font-display: swap to all @font-face rules
     * This is critical for preventing render-blocking font loading
     *
     * - Replaces font-display: block/auto with swap (these cause FOIT - Flash of Invisible Text)
     * - Adds font-display: swap to @font-face rules that don't have it
     *
     * @param string $css CSS content
     * @return string Modified CSS with font-display: swap
     */
    private function addFontDisplaySwap(string $css): string {
        // First, replace font-display: block or auto with swap
        // These values cause render-blocking behavior
        $css = preg_replace(
            '/font-display\s*:\s*(block|auto)\s*;?/i',
            'font-display:swap;',
            $css
        );

        // Then, add font-display: swap to @font-face rules that don't have it
        $css = preg_replace_callback(
            '/@font-face\s*\{([^}]+)\}/i',
            function($matches) {
                $rule = $matches[1];

                // If font-display already exists (swap, fallback, or optional), leave it
                if (stripos($rule, 'font-display') !== false) {
                    return $matches[0];
                }

                // Add font-display: swap at the beginning
                return '@font-face{font-display:swap;' . $rule . '}';
            },
            $css
        );

        return $css;
    }
    
    /**
     * Minify CSS file and save to cache
     *
     * @param string $file_path Path to CSS file
     * @return array Array with 'path' and 'url' of minified file
     * @throws Exception If file cannot be read or minified
     */
    public function minifyFile(string $file_path): array {
        if (!file_exists($file_path) || !is_readable($file_path)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Internal exception
            throw new Exception(sprintf('Cannot read CSS file: %s', esc_html($file_path)));
        }

        $content = prorank_read_file($file_path);
        if ($content === '') {
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Internal exception
            throw new Exception(sprintf('Failed to read CSS file: %s', esc_html($file_path)));
        }
        
        // Generate cache filename
        $file_hash = md5($file_path . filemtime($file_path));
        $cache_filename = $file_hash . '.min.css';
        $cache_path = $this->cache_dir . $cache_filename;
        $cache_url = $this->cache_url . $cache_filename;
        
        // Check if already cached
        if (file_exists($cache_path)) {
            // Use CssCacheServer for pretty URLs if available
            if (class_exists(CssCacheServer::class)) {
                $cache_url = CssCacheServer::get_cache_url($cache_filename);
            }
            
            return [
                'path' => $cache_path,
                'url' => $cache_url,
                'size_before' => filesize($file_path),
                'size_after' => filesize($cache_path),
            ];
        }
        
        // Minify content
        $minified = $this->minify($content, $file_path);
        
        // Save to cache
        $this->saveToCache($minified, $cache_path);
        
        // Create gzip version if enabled
        if ($this->options['gzip']) {
            $this->createGzipVersion($minified, $cache_path . '.gz');
        }
        
        // Use CssCacheServer for pretty URLs if available
        if (class_exists(CssCacheServer::class)) {
            $cache_url = CssCacheServer::get_cache_url($cache_filename);
        }
        
        return [
            'path' => $cache_path,
            'url' => $cache_url,
            'size_before' => strlen($content),
            'size_after' => strlen($minified),
            'compression_ratio' => round((1 - strlen($minified) / strlen($content)) * 100, 2),
        ];
    }
    
    /**
     * Combine multiple CSS files
     *
     * @param array $files Array of file paths
     * @return array Combined file info
     * @throws Exception If combination fails
     */
    public function combineFiles(array $files): array {
        $combined_content = '';
        $total_original_size = 0;
        
        foreach ($files as $file) {
            if (!file_exists($file) || !is_readable($file)) {
                continue;
            }
            
            $content = prorank_read_file($file);
            if ($content === '') {
                continue;
            }
            
            $total_original_size += strlen($content);
            
            // Add file marker for debugging
            $combined_content .= "\n/* ProRank CSS - Source: " . basename($file) . " */\n";
            $combined_content .= $content;
        }
        
        if (empty($combined_content)) {
            throw new Exception('No content to combine');
        }
        
        // Generate cache filename for combined file
        $files_hash = md5(implode('', $files) . $combined_content);
        $cache_filename = 'combined-' . $files_hash . '.min.css';
        $cache_path = $this->cache_dir . $cache_filename;
        $cache_url = $this->cache_url . $cache_filename;
        
        // Check if already cached
        if (file_exists($cache_path)) {
            return [
                'path' => $cache_path,
                'url' => $cache_url,
                'files_count' => count($files),
            ];
        }
        
        // Minify combined content
        $minified = $this->minify($combined_content);
        
        // Save to cache
        $this->saveToCache($minified, $cache_path);
        
        // Create gzip version if enabled
        if ($this->options['gzip']) {
            $this->createGzipVersion($minified, $cache_path . '.gz');
        }
        
        return [
            'path' => $cache_path,
            'url' => $cache_url,
            'files_count' => count($files),
            'size_before' => $total_original_size,
            'size_after' => strlen($minified),
            'compression_ratio' => round((1 - strlen($minified) / $total_original_size) * 100, 2),
        ];
    }
    
    /**
     * Clear CSS cache
     *
     * @param bool $force Force clear all files
     * @return int Number of files deleted
     */
    public function clearCache(bool $force = false): int {
        $deleted = 0;
        
        if (!is_dir($this->cache_dir)) {
            return 0;
        }
        
        $files = glob($this->cache_dir . '*.css');
        $gzip_files = glob($this->cache_dir . '*.css.gz');
        $all_files = array_merge($files ?: [], $gzip_files ?: []);
        
        foreach ($all_files as $file) {
            // Skip if not force and file is recent (less than 30 days old)
            if (!$force && (time() - filemtime($file) < 30 * DAY_IN_SECONDS)) {
                continue;
            }
            
            if (wp_delete_file($file)) {
                $deleted++;
            }
        }
        
        return $deleted;
    }
    
    /**
     * Get cache statistics
     *
     * @return array Cache statistics
     */
    public function getCacheStats(): array {
        if (!is_dir($this->cache_dir)) {
            return [
                'files' => 0,
                'size' => 0,
                'size_formatted' => '0 B',
            ];
        }
        
        $files = glob($this->cache_dir . '*.css');
        $gzip_files = glob($this->cache_dir . '*.css.gz');
        $all_files = array_merge($files ?: [], $gzip_files ?: []);
        
        $total_size = 0;
        foreach ($all_files as $file) {
            $total_size += filesize($file);
        }
        
        return [
            'files' => count($all_files),
            'size' => $total_size,
            'size_formatted' => size_format($total_size),
            'directory' => $this->cache_dir,
        ];
    }
    
    /**
     * Save content to cache file
     *
     * @param string $content Content to save
     * @param string $path    File path
     * @throws Exception If save fails
     */
    private function saveToCache(string $content, string $path): void {
        $result = file_put_contents($path, $content, LOCK_EX);
        
        if ($result === false) {
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Internal exception
            throw new Exception(sprintf('Failed to save minified CSS to cache: %s', esc_html($path)));
        }
    }
    
    /**
     * Create gzip version of file
     *
     * @param string $content Content to compress
     * @param string $path    Gzip file path
     */
    private function createGzipVersion(string $content, string $path): void {
        $compressed = gzencode($content, 9);
        
        if ($compressed !== false) {
            file_put_contents($path, $compressed, LOCK_EX);
        }
    }
    
    /**
     * Ensure cache directory exists
     */
    private function ensure_cache_directory(): void {
        if (!is_dir($this->cache_dir)) {
            wp_mkdir_p($this->cache_dir);
            
            // Add index.php for security
            $index_content = '<?php // Silence is golden';
            file_put_contents($this->cache_dir . 'index.php', $index_content);
        }
    }
}
