<?php
declare(strict_types=1);

/**
 * Module Manager
 *
 * @package ProRank\Core
 */

namespace ProRank\SEO\Core;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Contracts\ModuleInterface;

/**
 * Manages plugin modules
 */
class ModuleManager {
    
    /**
     * Registered modules
     *
     * @var array
     */
    private array $modules = [];
    
    /**
     * Settings manager instance
     *
     * @var SettingsManager
     */
    private SettingsManager $settings;
    
    
    /**
     * Constructor
     */
    public function __construct() {
        // Dependencies will be injected later
    }
    
    /**
     * Register a module
     *
     * @param ModuleInterface $module Module instance
     */
    public function register( ModuleInterface $module ): void {
        $this->modules[ $module->id() ] = $module;
    }
    
    /**
     * Register a module (alias for register)
     *
     * @param ModuleInterface $module Module instance
     */
    public function register_module( ModuleInterface $module ): void {
        $this->register( $module );
    }
    
    /**
     * Initialize active modules
     */
    public function init_active_modules(): void {
        // Ensure free modules are enabled by default on first run
        $settings = get_option( 'prorank_settings', [] );
        if ( ! is_array( $settings ) ) {
            $decoded = json_decode( (string) $settings, true );
            $settings = is_array( $decoded ) ? $decoded : [];
        }
        $active_modules = $settings['active_modules'] ?? [];
        if ( ! is_array( $active_modules ) ) {
            $active_modules = [];
        }

        if ( empty( $active_modules ) ) {
            foreach ( $this->modules as $module_id => $module ) {
                $active_modules[ $module_id ] = true;
                if ( method_exists( $module, 'get_slug' ) ) {
                    $active_modules[ $module->get_slug() ] = true;
                }
            }

            $settings['active_modules'] = $active_modules;
            $settings['modules_initialized'] = true;
            update_option( 'prorank_settings', $settings );
        }
        
        // Initialize all registered modules
        foreach ( $this->modules as $module_id => $module ) {
            if ( $this->is_active( $module_id ) ) {
                try {
                    $module->init();
                } catch (\Throwable $e) {
                    prorank_log(
                        sprintf(
                            '[ProRank SEO] Module init failed (%s, %s): %s',
                            $module_id,
                            is_object($module) ? get_class($module) : gettype($module),
                            $e->getMessage()
                        )
                    );
                }
            }
        }
    }
    
    /**
     * Get active modules
     *
     * @return array
     */
    public function get_active_modules(): array {
        $active = [];
        
        foreach ( $this->modules as $id => $module ) {
            if ( $this->is_active( $id ) ) {
                $active[ $id ] = $module;
            }
        }
        
        return $active;
    }
    
    /**
     * Check if module is active
     *
     * @param string $slug Module slug or ID
     * @return bool
     */
    public function is_active( string $slug ): bool {
        // Find the module - could be by ID or slug
        $module = null;
        if ( isset( $this->modules[ $slug ] ) ) {
            $module = $this->modules[ $slug ];
        } else {
            // Try to find by slug property
            foreach ( $this->modules as $module_id => $mod ) {
                if ( method_exists( $mod, 'get_slug' ) && $mod->get_slug() === $slug ) {
                    $module = $mod;
                    break;
                }
            }
        }
        
        if ( ! $module ) {
            return false;
        }
        
        // Then check settings for active modules
        $settings = get_option( 'prorank_settings', [] );
        if (!is_array($settings)) {
            $decoded = json_decode((string) $settings, true);
            $settings = is_array($decoded) ? $decoded : [];
        }
        $active_modules = $settings['active_modules'] ?? [];
        if (!is_array($active_modules)) {
            $active_modules = [];
        }

        // If no modules have been configured yet, enable all by default
        if ( empty( $active_modules ) ) {
            return true;
        }

        // Check if module is explicitly disabled (key exists but value is false/empty)
        // This handles the case where user has toggled off a module
        $module_slug = method_exists( $module, 'get_slug' ) ? $module->get_slug() : null;

        // Check if explicitly set to false (disabled)
        if ( array_key_exists( $slug, $active_modules ) && empty( $active_modules[ $slug ] ) ) {
            return false;
        }
        if ( $module_slug && array_key_exists( $module_slug, $active_modules ) && empty( $active_modules[ $module_slug ] ) ) {
            return false;
        }

        // Check both by the provided slug and the module's slug property
        if ( ! empty( $active_modules[ $slug ] ) ) {
            return true;
        }

        if ( $module_slug && ! empty( $active_modules[ $module_slug ] ) ) {
            return true;
        }

        // Default-enable free modules when not explicitly disabled.
        if ( method_exists( $module, 'get_feature_tier' ) && $module->get_feature_tier() === 'free' ) {
            return true;
        }

        // Performance modules default to enabled if tier allows and not explicitly disabled
        if ( method_exists( $module, 'get_parent_slug' ) && $module->get_parent_slug() === 'performance' ) {
            return true;
        }

        return false;
    }
    
    /**
     * Check if module is enabled (alias for is_active)
     *
     * @param string $slug Module slug or ID
     * @return bool
     */
    public function is_module_enabled( string $slug ): bool {
        return $this->is_active( $slug );
    }
    
    /**
     * Get all registered modules
     *
     * @return array
     */
    public function get_modules(): array {
        return $this->modules;
    }
    
    /**
     * Check if a module is registered
     *
     * @param string $slug Module slug or ID
     * @return bool
     */
    public function has_module( string $slug ): bool {
        // Check direct lookup
        if ( isset( $this->modules[ $slug ] ) ) {
            return true;
        }
        
        // Try to find by slug property
        foreach ( $this->modules as $module ) {
            if ( method_exists( $module, 'get_slug' ) && $module->get_slug() === $slug ) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get a specific module
     *
     * @param string $slug Module slug
     * @return ModuleInterface|null
     */
    public function get_module( string $slug ): ?ModuleInterface {
        return $this->modules[ $slug ] ?? null;
    }
    
    /**
     * Enable a module
     *
     * @param string $slug Module slug or ID
     * @return bool
     */
    public function enable_module( string $slug ): bool {
        $settings = get_option( 'prorank_settings', [] );
        if (!is_array($settings)) {
            $decoded = json_decode((string) $settings, true);
            $settings = is_array($decoded) ? $decoded : [];
        }
        $active_modules = $settings['active_modules'] ?? [];
        if (!is_array($active_modules)) {
            $active_modules = [];
        }

        // Enable by the provided slug
        $active_modules[ $slug ] = true;

        // Also enable by module ID if we can find it
        foreach ( $this->modules as $module_id => $module ) {
            if ( method_exists( $module, 'get_slug' ) && $module->get_slug() === $slug ) {
                $active_modules[ $module_id ] = true;
                break;
            }
        }

        $settings['active_modules'] = $active_modules;
        // update_option returns false if value unchanged, but that's still success
        update_option( 'prorank_settings', $settings );
        return true;
    }
    
    /**
     * Disable a module
     *
     * @param string $slug Module slug or ID
     * @return bool
     */
    public function disable_module( string $slug ): bool {
        $settings = get_option( 'prorank_settings', [] );
        if (!is_array($settings)) {
            $decoded = json_decode((string) $settings, true);
            $settings = is_array($decoded) ? $decoded : [];
        }
        $active_modules = $settings['active_modules'] ?? [];
        if (!is_array($active_modules)) {
            $active_modules = [];
        }

        // Set to false to explicitly mark as disabled (don't just unset)
        $active_modules[ $slug ] = false;

        // Also disable by module ID if we can find it
        foreach ( $this->modules as $module_id => $module ) {
            if ( method_exists( $module, 'get_slug' ) && $module->get_slug() === $slug ) {
                $active_modules[ $module_id ] = false;
            }
        }

        $settings['active_modules'] = $active_modules;
        // update_option returns false if value unchanged, but that's still success
        update_option( 'prorank_settings', $settings );
        return true;
    }

    /**
     * Activate a module (alias for enable_module)
     *
     * @param string $slug Module slug
     * @return bool
     */
    public function activate_module( string $slug ): bool {
        return $this->enable_module( $slug );
    }
    
    /**
     * Deactivate a module (alias for disable_module)
     *
     * @param string $slug Module slug
     * @return bool
     */
    public function deactivate_module( string $slug ): bool {
        return $this->disable_module( $slug );
    }
    
    /**
     * Get top-level modules (modules without parent)
     *
     * @return array
     */
    public function get_top_level_modules(): array {
        $top_level = [];
        
        foreach ( $this->modules as $slug => $module ) {
            if ( $module->get_parent_slug() === null ) {
                $top_level[ $slug ] = $module;
            }
        }
        
        return $top_level;
    }
    
    /**
     * Get child modules of a parent module
     *
     * @param string $parent_slug Parent module slug
     * @return array
     */
    public function get_child_modules( string $parent_slug ): array {
        $children = [];
        
        foreach ( $this->modules as $slug => $module ) {
            if ( $module->get_parent_slug() === $parent_slug ) {
                $children[ $slug ] = $module;
            }
        }
        
        return $children;
    }

    // Premium module registration lives in the premium plugin.
}
