<?php
/**
 * Migration Manager
 *
 * Handles plugin version upgrades and data migrations.
 *
 * @package ProRank\SEO\Core
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Migration class
 */
class Migration {

    /**
     * Settings manager instance
     *
     * @var SettingsManager
     */
    private SettingsManager $settings_manager;


    /**
     * Option key for storing plugin version
     */
    private const VERSION_OPTION = 'prorank_seo_version';

    /**
     * Option key for storing migration history
     */
    private const MIGRATION_HISTORY_OPTION = 'prorank_seo_migration_history';

    /**
     * Constructor
     *
     * @param SettingsManager $settings_manager Settings manager instance.
     */
    public function __construct(SettingsManager $settings_manager) {
        $this->settings_manager = $settings_manager;
    }

    /**
     * Initialize migration system
     *
     * @return void
     */
    public function init(): void {
        // Check for version changes on admin_init
        add_action( 'admin_init', [ $this, 'check_version' ] );
    }

    /**
     * Check if migration is needed
     *
     * @return void
     */
    public function check_version(): void {
        $stored_version  = get_option( self::VERSION_OPTION, '0.0.0' );
        $current_version = PRORANK_VERSION;

        // Skip if versions match
        if ( version_compare( $stored_version, $current_version, '=' ) ) {
            return;
        }

        // Run migrations
        $this->run_migrations( $stored_version, $current_version );

        // Update stored version
        update_option( self::VERSION_OPTION, $current_version );
    }

    /**
     * Run migrations between versions
     *
     * @param string $from_version Previous version.
     * @param string $to_version   Current version.
     * @return void
     */
    private function run_migrations( string $from_version, string $to_version ): void {
        // Get migration history
        $history = get_option( self::MIGRATION_HISTORY_OPTION, [] );

        /**
         * Define migrations
         * Each migration has a version it applies to and a callback
         */
        $migrations = [
            // Example: Settings structure change in v0.3.0
            '0.3.0' => [ $this, 'migrate_0_3_0_settings_structure' ],
        ];

        foreach ( $migrations as $version => $callback ) {
            // Skip if this version is not in our upgrade path
            if ( version_compare( $from_version, $version, '>=' ) ) {
                continue;
            }
            if ( version_compare( $to_version, $version, '<' ) ) {
                continue;
            }

            // Skip if already run
            $migration_key = 'migration_' . str_replace( '.', '_', $version );
            if ( isset( $history[ $migration_key ] ) ) {
                continue;
            }

            // Run migration
            try {
                call_user_func( $callback );
                
                // Record success
                $history[ $migration_key ] = [
                    'version'     => $version,
                    'executed_at' => time(),
                    'status'      => 'success',
                ];
                
                // Log migration
                $this->log_migration( "Successfully ran migration for version {$version}" );
                
            } catch ( \Exception $e ) {
                // Record failure
                $history[ $migration_key ] = [
                    'version'     => $version,
                    'executed_at' => time(),
                    'status'      => 'failed',
                    'error'       => esc_html($e->getMessage()),
                ];
                
                // Log error
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    prorank_log( "ProRank SEO Migration Error ({$version}): " . $e->getMessage() );
                }
            }
        }

        // Update history
        update_option( self::MIGRATION_HISTORY_OPTION, $history );
    }

    /**
     * Example migration for v0.3.0 settings structure
     *
     * @return void
     */
    private function migrate_0_3_0_settings_structure(): void {
        // Example of migrating old settings structure to new
        $old_settings = get_option( 'prorank_old_settings', [] );
        
        if ( ! empty( $old_settings ) ) {
            // Map old settings to new structure
            foreach ( $old_settings as $key => $value ) {
                // Determine module and setting key
                $module = $this->determine_module_for_setting( $key );
                $new_key = $this->transform_setting_key( $key );
                
                // Update using settings manager
                $this->settings_manager->update_setting( $module, $new_key, $value );
            }
            
            // Remove old option
            delete_option( 'prorank_old_settings' );
        }
    }

    /**
     * Determine module for old setting key
     *
     * @param string $old_key Old setting key.
     * @return string Module slug.
     */
    private function determine_module_for_setting( string $old_key ): string {
        // Map old setting prefixes to modules
        $prefix_map = [
            'seo_title_'    => 'titles-meta',
            'sitemap_'      => 'sitemaps',
            'redirect_'     => 'redirects',
            'cache_'        => 'page-caching',
            'schema_'       => 'schema-markup',
            'social_'       => 'social',
            'analytics_'    => 'analytics',
        ];

        foreach ( $prefix_map as $prefix => $module ) {
            if ( str_starts_with( $old_key, $prefix ) ) {
                return $module;
            }
        }

        return 'general';
    }

    /**
     * Transform old setting key to new format
     *
     * @param string $old_key Old setting key.
     * @return string New setting key.
     */
    private function transform_setting_key( string $old_key ): string {
        // Remove module prefix if present
        $prefixes = [ 'seo_title_', 'sitemap_', 'redirect_', 'cache_', 'schema_', 'social_', 'analytics_' ];
        
        foreach ( $prefixes as $prefix ) {
            if ( str_starts_with( $old_key, $prefix ) ) {
                return substr( $old_key, strlen( $prefix ) );
            }
        }

        return $old_key;
    }

    /**
     * Get feature display name
     *
     * @param string $module_slug  Module slug.
     * @param string $feature_slug Feature slug.
     * @return string Display name.
     */
    private function get_feature_display_name( string $module_slug, string $feature_slug ): string {
        // In a real implementation, this would look up proper display names
        $feature_names = [
            'head_cleanup'      => __( 'Head Cleanup', 'prorank-seo' ),
            'content_safeguard' => __( 'Content Safeguard', 'prorank-seo' ),
            'global_defaults'   => __( 'Global Defaults', 'prorank-seo' ),
        ];

        return $feature_names[ $feature_slug ] ?? ucwords( str_replace( [ '-', '_' ], ' ', $feature_slug ) );
    }

    /**
     * Log migration message
     *
     * @param string $message Message to log.
     * @return void
     */
    private function log_migration( string $message ): void {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log( 'ProRank SEO Migration: ' . $message );
        }
    }

    /**
     * Get migration history
     *
     * @return array Migration history.
     */
    public function get_migration_history(): array {
        return get_option( self::MIGRATION_HISTORY_OPTION, [] );
    }

    /**
     * Clear migration history (for testing)
     *
     * @return void
     */
    public function clear_migration_history(): void {
        delete_option( self::MIGRATION_HISTORY_OPTION );
    }
}
