<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries throughout
/**
 * Link Click Tracker
 * 
 * Tracks clicks on internal links for analytics
 */

declare(strict_types=1);

namespace ProRank\SEO\Core;

defined( 'ABSPATH' ) || exit;

/**
 * LinkClickTracker class
 */
class LinkClickTracker {
    
    /**
     * Database table name
     * 
     * @var string
     */
    private string $table_name;
    
    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'prorank_link_clicks';
    }
    
    /**
     * Initialize the tracker
     */
    public static function init(): void {
        if ( function_exists( 'prorank_is_click_tracking_enabled' ) && ! prorank_is_click_tracking_enabled() ) {
            return;
        }
        $instance = new self();
        
        // Add hooks
        add_action('wp_enqueue_scripts', [$instance, 'enqueue_tracking_script']);
        add_action('wp_ajax_prorank_track_link_click', [$instance, 'track_click']);
        add_action('wp_ajax_nopriv_prorank_track_link_click', [$instance, 'track_click']);
        
        // Create table on activation
        register_activation_hook(PRORANK_PLUGIN_FILE, [$instance, 'create_table']);
    }
    
    /**
     * Create database table for link clicks
     */
    public function create_table(): void {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            source_id bigint(20) unsigned NOT NULL,
            target_id bigint(20) unsigned DEFAULT NULL,
            target_url text NOT NULL,
            click_date datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            user_id bigint(20) unsigned DEFAULT NULL,
            session_id varchar(32) DEFAULT NULL,
            PRIMARY KEY (id),
            KEY source_id (source_id),
            KEY target_id (target_id),
            KEY click_date (click_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Enqueue tracking script
     */
    public function enqueue_tracking_script(): void {
        if ( function_exists( 'prorank_is_click_tracking_enabled' ) && ! prorank_is_click_tracking_enabled() ) {
            return;
        }
        if (!is_singular()) {
            return;
        }
        
        wp_enqueue_script(
            'prorank-link-tracker',
            PRORANK_PLUGIN_URL . 'assets/js/link-tracker.js',
            ['jquery'],
            PRORANK_VERSION,
            true
        );
        
        wp_localize_script('prorank-link-tracker', 'prorankLinkTracker', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('prorank_link_tracker'),
            'postId' => get_the_ID(),
            'siteUrl' => get_site_url(),
        ]);
    }
    
    /**
     * Track link click via AJAX
     */
    public function track_click(): void {
        if ( function_exists( 'prorank_is_click_tracking_enabled' ) && ! prorank_is_click_tracking_enabled() ) {
            wp_send_json_error( __( 'Click tracking is disabled.', 'prorank-seo' ) );
            return;
        }
        if (!check_ajax_referer('prorank_link_tracker', 'nonce', false)) {
            wp_send_json_error( __( 'Invalid nonce.', 'prorank-seo' ), 403 );
            return;
        }

        $source_id = isset($_POST['source_id']) ? absint( wp_unslash( $_POST['source_id'] ) ) : 0;
        $target_url = isset($_POST['target_url']) ? esc_url_raw( wp_unslash( $_POST['target_url'] ) ) : '';
        
        if (!$source_id || !$target_url) {
            wp_send_json_error('Invalid parameters');
            return;
        }
        
        // Get target post ID if internal link
        $target_id = url_to_postid($target_url);
        
        // Get user ID and session
        $user_id = get_current_user_id();
        $session_id = session_id() ?: wp_generate_password(32, false);
        
        // Insert click record
        global $wpdb;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $result = $wpdb->insert(
            $this->table_name,
            [
                'source_id' => $source_id,
                'target_id' => $target_id ?: null,
                'target_url' => $target_url,
                'user_id' => $user_id ?: null,
                'session_id' => $session_id,
                'click_date' => current_time('mysql'),
            ],
            ['%d', '%d', '%s', '%d', '%s', '%s']
        );
        
        if ($result === false) {
            wp_send_json_error('Failed to track click');
            return;
        }
        
        // Update daily stats
        $this->update_daily_stats();
        
        wp_send_json_success('Click tracked');
    }
    
    /**
     * Update daily stats
     */
    private function update_daily_stats(): void {
        global $wpdb;
        
        // Get today's click count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $today_clicks = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$this->table_name}
            WHERE DATE(click_date) = CURDATE()
        ");
        
        // Get yesterday's click count
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $yesterday_clicks = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$this->table_name}
            WHERE DATE(click_date) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)
        ");
        
        // Calculate change percentage
        $change = 0;
        if ($yesterday_clicks > 0) {
            $change = (($today_clicks - $yesterday_clicks) / $yesterday_clicks) * 100;
        }
        
        // Update options
        update_option('prorank_link_clicks', $today_clicks);
        update_option('prorank_link_clicks_change', round($change));
    }
    
    /**
     * Get click stats for a specific period
     * 
     * @param int $days Number of days to look back
     * @return array Click statistics
     */
    public function get_click_stats(int $days = 30): array {
        global $wpdb;
        
        // Total clicks in period
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $total_clicks = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) 
            FROM {$this->table_name}
            WHERE click_date >= DATE_SUB(NOW(), INTERVAL %d DAY)
        ", $days));
        
        // Clicks by day
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $daily_clicks = $wpdb->get_results($wpdb->prepare("
            SELECT DATE(click_date) as date, COUNT(*) as clicks
            FROM {$this->table_name}
            WHERE click_date >= DATE_SUB(NOW(), INTERVAL %d DAY)
            GROUP BY DATE(click_date)
            ORDER BY date DESC
        ", $days), ARRAY_A);
        
        // Most clicked links
        $top_links = $wpdb->get_results($wpdb->prepare("
            SELECT 
                target_url,
                target_id,
                COUNT(*) as click_count,
                p.post_title
            FROM {$this->table_name} c
            LEFT JOIN {$wpdb->posts} p ON c.target_id = p.ID
            WHERE c.click_date >= DATE_SUB(NOW(), INTERVAL %d DAY)
            GROUP BY target_url, target_id
            ORDER BY click_count DESC
            LIMIT 10
        ", $days), ARRAY_A);
        
        // Calculate trend
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $first_half = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) 
            FROM {$this->table_name}
            WHERE click_date >= DATE_SUB(NOW(), INTERVAL %d DAY)
            AND click_date < DATE_SUB(NOW(), INTERVAL %d DAY)
        ", $days, $days / 2));
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $second_half = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) 
            FROM {$this->table_name}
            WHERE click_date >= DATE_SUB(NOW(), INTERVAL %d DAY)
        ", $days / 2));
        
        $trend = 0;
        if ($first_half > 0) {
            $trend = (($second_half - $first_half) / $first_half) * 100;
        }
        
        return [
            'total_clicks' => (int) $total_clicks,
            'daily_clicks' => $daily_clicks,
            'top_links' => $top_links,
            'trend' => round($trend, 1),
            'period_days' => $days,
        ];
    }
    
    /**
     * Get clicks for a specific post
     * 
     * @param int $post_id Post ID
     * @return array Click data
     */
    public function get_post_clicks(int $post_id): array {
        global $wpdb;
        
        // Outbound clicks from this post
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $outbound_clicks = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) 
            FROM {$this->table_name}
            WHERE source_id = %d
        ", $post_id));
        
        // Inbound clicks to this post
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $inbound_clicks = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) 
            FROM {$this->table_name}
            WHERE target_id = %d
        ", $post_id));
        
        return [
            'outbound' => (int) $outbound_clicks,
            'inbound' => (int) $inbound_clicks,
            'total' => (int) ($outbound_clicks + $inbound_clicks),
        ];
    }
}

// Initialize the tracker
add_action('init', [LinkClickTracker::class, 'init']);
