<?php
/**
 * License Manager - Free Version Stub
 *
 * This is a simplified LicenseManager for the free version.
 * Always returns free tier without any license validation.
 *
 * @package ProRank\SEO\Core
 */

namespace ProRank\SEO\Core;

defined( 'ABSPATH' ) || exit;

/**
 * License Manager class - Free version stub
 */
class LicenseManager {

    // Tier constants
    const TIER_FREE = 'free';
    const TIER_PRO = 'pro';
    const TIER_BUSINESS = 'business';
    const TIER_AGENCY = 'agency';

    /**
     * Singleton instance
     */
    private static $instance = null;

    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    public function __construct() {
        // No initialization needed for free version
    }

    /**
     * Get current tier - always free
     */
    public function get_tier() {
        return self::TIER_FREE;
    }

    /**
     * Get current tier - alias
     */
    public function get_current_tier() {
        return self::TIER_FREE;
    }

    /**
     * Check if tier is active - only free is active
     */
    public function is_tier_active($tier) {
        return $tier === self::TIER_FREE;
    }

    /**
     * Alias for tier check (compat with premium code).
     */
    public function is_tier($tier) {
        return $this->is_tier_active($tier);
    }

    /**
     * Check if license is valid - always true for free
     */
    public function is_valid() {
        return true;
    }

    /**
     * Check if license is active - always true for free
     */
    public function is_active() {
        return true;
    }

    /**
     * Get license status
     */
    public function get_status() {
        return 'active';
    }

    /**
     * Check if in developer mode - always false for free
     */
    public function is_developer_mode() {
        return false;
    }

    /**
     * Legacy alias used in some admin helpers.
     *
     * @return bool
     */
    public function is_developer_mode_active() {
        return false;
    }

    /**
     * Get license key - empty for free
     */
    public function get_license_key() {
        $key = get_option('prorank_seo_license_key', '');
        if (!empty($key)) {
            return $key;
        }
        return get_option('prorank_license_key', '');
    }

    /**
     * Activate license - no-op for free
     */
    public function activate($license_key) {
        return false;
    }

    /**
     * Deactivate license - no-op for free
     */
    public function deactivate() {
        return false;
    }

    /**
     * Check license - no-op for free
     */
    public function check_license() {
        return true;
    }

    /**
     * Get license data
     */
    public function get_license_data() {
        return [
            'tier' => self::TIER_FREE,
            'status' => 'active',
            'is_valid' => true,
        ];
    }

    /**
     * License status snapshot (compat for modules relying on cached status).
     *
     * @return array<string, mixed>
     */
    public function get_license_status_snapshot(): array {
        return [
            'tier' => self::TIER_FREE,
            'status' => 'active',
            'is_valid' => true,
            'expires_at' => null,
        ];
    }

    /**
     * Whether the current license snapshot is valid.
     *
     * @return bool
     */
    public function has_valid_license_status(): bool {
        return true;
    }

    /**
     * Get list of active tiers.
     *
     * @return array<int, string>
     */
    public function get_active_tiers(): array {
        return [self::TIER_FREE];
    }

    /**
     * Check tier access (compat helper).
     *
     * @param string $tier
     * @return bool
     */
    public function has_tier_access(string $tier): bool {
        return $this->is_tier_active($tier);
    }

    /**
     * Get display name for a tier.
     *
     * @param string $tier
     * @return string
     */
    public function get_tier_display_name(string $tier): string {
        switch ($tier) {
            case self::TIER_PRO:
                return 'Pro';
            case self::TIER_BUSINESS:
                return 'Business';
            case self::TIER_AGENCY:
                return 'Agency';
            case self::TIER_FREE:
            default:
                return 'Free';
        }
    }

    /**
     * JWT token for API calls (not used in free).
     *
     * @return string
     */
    public function get_jwt_token(): string {
        return '';
    }

    /**
     * Reset and reactivate license (no-op in free).
     *
     * @return bool
     */
    public function reset_and_reactivate(): bool {
        return true;
    }
}
