<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * Yoast SEO Importer
 *
 * @package ProRank\SEO\Core\Importers
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Importers;

defined( 'ABSPATH' ) || exit;

/**
 * YoastImporter class
 * 
 * Imports data from Yoast SEO plugin
 */
class YoastImporter extends BaseImporter {
    
    /**
     * Yoast meta key mappings
     *
     * @var array
     */
    private array $meta_key_map = [
        '_yoast_wpseo_title' => '_prorank_seo_title',
        '_yoast_wpseo_metadesc' => '_prorank_seo_description',
        '_yoast_wpseo_focuskw' => '_prorank_seo_focus_keyword',
        '_yoast_wpseo_canonical' => '_prorank_seo_canonical',
        '_yoast_wpseo_meta-robots-noindex' => '_prorank_seo_noindex',
        '_yoast_wpseo_meta-robots-nofollow' => '_prorank_seo_nofollow',
        '_yoast_wpseo_opengraph-title' => '_prorank_seo_og_title',
        '_yoast_wpseo_opengraph-description' => '_prorank_seo_og_description',
        '_yoast_wpseo_opengraph-image' => '_prorank_seo_og_image',
        '_yoast_wpseo_twitter-title' => '_prorank_seo_twitter_title',
        '_yoast_wpseo_twitter-description' => '_prorank_seo_twitter_description',
        '_yoast_wpseo_twitter-image' => '_prorank_seo_twitter_image',
    ];
    
    /**
     * Check if Yoast is available
     *
     * @return bool
     */
    public function is_available(): bool {
        return defined('WPSEO_VERSION') || 
               class_exists('WPSEO_Options') || 
               $this->has_yoast_data();
    }
    
    /**
     * Check if Yoast data exists in database
     *
     * @return bool
     */
    private function has_yoast_data(): bool {
        global $wpdb;
        
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $result = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} 
             WHERE meta_key LIKE '_yoast_wpseo_%' 
             LIMIT 1"
        );
        
        return (int) $result > 0;
    }
    
    /**
     * Run the import
     *
     * @return array
     */
    public function import(): array {
        if (!$this->is_available()) {
            return [
                'success' => false,
                'message' => __('Yoast SEO data not found', 'prorank-seo'),
                'stats' => $this->stats,
            ];
        }
        
        try {
            // Import post/page meta
            $this->import_post_meta();
            
            // Import term meta
            $this->import_term_meta();
            
            // Import settings
            $this->import_settings();
            
            // Import redirects if available
            $this->import_redirects();
            
            // Import schema settings
            $this->import_schema();
            
            return [
                'success' => true,
                'message' => __('Yoast SEO data imported successfully', 'prorank-seo'),
                'stats' => $this->stats,
            ];
            
        } catch (\Exception $e) {
            $this->log_error('Import failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => esc_html($e->getMessage()),
                'stats' => $this->stats,
            ];
        }
    }
    
    /**
     * Import single post meta
     *
     * @param int $post_id Post ID
     * @return void
     */
    protected function import_single_post_meta(int $post_id): void {
        foreach ($this->meta_key_map as $yoast_key => $prorank_key) {
            $value = get_post_meta($post_id, $yoast_key, true);
            
            if (!empty($value)) {
                // Special handling for robots meta
                if (strpos($yoast_key, 'noindex') !== false) {
                    $value = ($value === '1') ? 'noindex' : 'index';
                }
                if (strpos($yoast_key, 'nofollow') !== false) {
                    $value = ($value === '1') ? 'nofollow' : 'follow';
                }
                
                update_post_meta($post_id, $prorank_key, $this->sanitize_meta_value($value));
            }
        }
        
        // Import primary category
        $primary_category = get_post_meta($post_id, '_yoast_wpseo_primary_category', true);
        if ($primary_category) {
            update_post_meta($post_id, '_prorank_seo_primary_category', $primary_category);
        }
        
        // Import breadcrumb title
        $breadcrumb_title = get_post_meta($post_id, '_yoast_wpseo_bctitle', true);
        if ($breadcrumb_title) {
            update_post_meta($post_id, '_prorank_seo_breadcrumb_title', $breadcrumb_title);
        }
    }
    
    /**
     * Import single term meta
     *
     * @param int $term_id Term ID
     * @return void
     */
    protected function import_single_term_meta(int $term_id): void {
        // Get Yoast term meta from options table (older versions)
        $term_meta = get_option('wpseo_taxonomy_meta');
        
        if ($term_meta && is_array($term_meta)) {
            foreach ($term_meta as $taxonomy => $terms) {
                if (isset($terms[$term_id])) {
                    $meta = $terms[$term_id];
                    
                    // Map Yoast term meta to ProRank
                    if (isset($meta['wpseo_title'])) {
                        update_term_meta($term_id, '_prorank_seo_title', $meta['wpseo_title']);
                    }
                    if (isset($meta['wpseo_desc'])) {
                        update_term_meta($term_id, '_prorank_seo_description', $meta['wpseo_desc']);
                    }
                    if (isset($meta['wpseo_canonical'])) {
                        update_term_meta($term_id, '_prorank_seo_canonical', $meta['wpseo_canonical']);
                    }
                    if (isset($meta['wpseo_noindex'])) {
                        $noindex = $meta['wpseo_noindex'] === 'noindex' ? 'noindex' : 'index';
                        update_term_meta($term_id, '_prorank_seo_noindex', $noindex);
                    }
                }
            }
        }
        
        // Also check for newer term meta format
        $title = get_term_meta($term_id, 'wpseo_title', true);
        if ($title) {
            update_term_meta($term_id, '_prorank_seo_title', $title);
        }
        
        $desc = get_term_meta($term_id, 'wpseo_desc', true);
        if ($desc) {
            update_term_meta($term_id, '_prorank_seo_description', $desc);
        }
    }
    
    /**
     * Import Yoast settings
     *
     * @return void
     */
    protected function import_settings(): void {
        // Import general settings
        $wpseo = get_option('wpseo');
        if ($wpseo && is_array($wpseo)) {
            $settings = [];
            
            // Map separator
            if (isset($wpseo['separator'])) {
                $separator_map = [
                    'sc-dash' => '-',
                    'sc-ndash' => '–',
                    'sc-mdash' => '—',
                    'sc-colon' => ':',
                    'sc-middot' => '·',
                    'sc-bull' => '•',
                    'sc-star' => '*',
                    'sc-smstar' => '⋆',
                    'sc-pipe' => '|',
                    'sc-tilde' => '~',
                ];
                $settings['title_separator'] = $separator_map[$wpseo['separator']] ?? '-';
            }
            
            // Save settings
            if (!empty($settings)) {
                update_option('prorank_seo_settings', $settings);
                $this->stats['settings']++;
            }
        }
        
        // Import social settings
        $social = get_option('wpseo_social');
        if ($social && is_array($social)) {
            $social_settings = [];
            
            if (isset($social['facebook_site'])) {
                $social_settings['facebook_page_url'] = $social['facebook_site'];
            }
            if (isset($social['twitter_site'])) {
                $social_settings['twitter_username'] = str_replace('@', '', $social['twitter_site']);
            }
            
            if (!empty($social_settings)) {
                update_option('prorank_seo_social', $social_settings);
                $this->stats['settings']++;
            }
        }
    }
    
    /**
     * Import Yoast Premium redirects
     *
     * @return void
     */
    protected function import_redirects(): void {
        global $wpdb;
        
        // Check if Yoast Premium redirects table exists
        $table = $wpdb->prefix . 'yoast_seo_redirects';
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) !== $table) {
            return;
        }
        
        $redirects = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM %i WHERE status = %s",
            $table,
            'enabled'
        ));
        
        foreach ($redirects as $redirect) {
            // Store redirect in ProRank format
            $prorank_redirect = [
                'from' => $redirect->origin,
                'to' => $redirect->target,
                'type' => $redirect->type,
                'created' => current_time('mysql'),
            ];
            
            // Save to ProRank redirects (implement your redirect storage)
            $this->save_redirect($prorank_redirect);
            $this->stats['redirects']++;
        }
    }
    
    /**
     * Import schema settings
     *
     * @return void
     */
    protected function import_schema(): void {
        $schema = get_option('wpseo_schema');
        
        if ($schema && is_array($schema)) {
            // Map Yoast schema settings to ProRank
            $prorank_schema = [];
            
            if (isset($schema['company_or_person'])) {
                $prorank_schema['site_representation'] = $schema['company_or_person'];
            }
            
            if (isset($schema['company_name'])) {
                $prorank_schema['organization_name'] = $schema['company_name'];
            }
            
            if (isset($schema['company_logo'])) {
                $prorank_schema['organization_logo'] = $schema['company_logo'];
            }
            
            if (!empty($prorank_schema)) {
                update_option('prorank_seo_schema', $prorank_schema);
                $this->stats['settings']++;
            }
        }
    }
    
    /**
     * Map meta key
     *
     * @param string $competitor_key Yoast meta key
     * @return string ProRank meta key
     */
    protected function map_meta_key(string $competitor_key): string {
        return $this->meta_key_map[$competitor_key] ?? $competitor_key;
    }
    
    /**
     * Save redirect (placeholder - implement based on your redirect system)
     *
     * @param array $redirect Redirect data
     * @return void
     */
    private function save_redirect(array $redirect): void {
        // This would integrate with your redirect manager
        // For now, store in options
        $redirects = get_option('prorank_seo_redirects', []);
        $redirects[] = $redirect;
        update_option('prorank_seo_redirects', $redirects);
    }
}
