<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * Rank Math SEO Importer
 *
 * @package ProRank\SEO\Core\Importers
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\Importers;

defined( 'ABSPATH' ) || exit;

/**
 * RankMathImporter class
 *
 * Imports data from Rank Math SEO plugin
 */
class RankMathImporter extends BaseImporter {

    /**
     * Rank Math meta key mappings
     *
     * @var array
     */
    private array $meta_key_map = [
        'rank_math_title' => '_prorank_seo_title',
        'rank_math_description' => '_prorank_seo_description',
        'rank_math_focus_keyword' => '_prorank_seo_focus_keyword',
        'rank_math_canonical_url' => '_prorank_seo_canonical',
        'rank_math_robots' => '_prorank_seo_robots',
        'rank_math_facebook_title' => '_prorank_seo_og_title',
        'rank_math_facebook_description' => '_prorank_seo_og_description',
        'rank_math_facebook_image' => '_prorank_seo_og_image',
        'rank_math_twitter_title' => '_prorank_seo_twitter_title',
        'rank_math_twitter_description' => '_prorank_seo_twitter_description',
        'rank_math_twitter_image' => '_prorank_seo_twitter_image',
        'rank_math_primary_category' => '_prorank_seo_primary_category',
        'rank_math_breadcrumb_title' => '_prorank_seo_breadcrumb_title',
    ];

    /**
     * Check if Rank Math is available
     *
     * @return bool
     */
    public function is_available(): bool {
        return defined('RANK_MATH_VERSION') ||
               class_exists('RankMath') ||
               $this->has_rankmath_data();
    }

    /**
     * Check if Rank Math data exists in database
     *
     * @return bool
     */
    private function has_rankmath_data(): bool {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $result = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta}
             WHERE meta_key LIKE 'rank_math_%'
             LIMIT 1"
        );

        return (int) $result > 0;
    }

    /**
     * Run the import
     *
     * @return array
     */
    public function import(): array {
        if (!$this->is_available()) {
            return [
                'success' => false,
                'message' => __('Rank Math SEO data not found', 'prorank-seo'),
                'stats' => $this->stats,
            ];
        }

        try {
            // Import post/page meta
            $this->import_post_meta();

            // Import term meta
            $this->import_term_meta();

            // Import settings
            $this->import_settings();

            // Import redirects
            $this->import_redirects();

            // Import schema settings
            $this->import_schema();

            return [
                'success' => true,
                'message' => __('Rank Math SEO data imported successfully', 'prorank-seo'),
                'stats' => $this->stats,
            ];

        } catch (\Exception $e) {
            $this->log_error('Import failed: ' . $e->getMessage());

            return [
                'success' => false,
                'message' => esc_html($e->getMessage()),
                'stats' => $this->stats,
            ];
        }
    }

    /**
     * Import single post meta
     *
     * @param int $post_id Post ID
     * @return void
     */
    protected function import_single_post_meta(int $post_id): void {
        foreach ($this->meta_key_map as $rm_key => $prorank_key) {
            $value = get_post_meta($post_id, $rm_key, true);

            if (!empty($value)) {
                // Handle robots meta array
                if ($rm_key === 'rank_math_robots' && is_array($value)) {
                    $robots = [];
                    if (in_array('noindex', $value, true)) {
                        $robots[] = 'noindex';
                        update_post_meta($post_id, '_prorank_seo_noindex', 'noindex');
                    }
                    if (in_array('nofollow', $value, true)) {
                        $robots[] = 'nofollow';
                        update_post_meta($post_id, '_prorank_seo_nofollow', 'nofollow');
                    }
                    continue;
                }

                update_post_meta($post_id, $prorank_key, $this->sanitize_meta_value($value));
            }
        }

        // Import SEO score
        $seo_score = get_post_meta($post_id, 'rank_math_seo_score', true);
        if ($seo_score) {
            update_post_meta($post_id, '_prorank_seo_score', intval($seo_score));
        }

        // Import schema data
        $schema = get_post_meta($post_id, 'rank_math_rich_snippet', true);
        if ($schema) {
            update_post_meta($post_id, '_prorank_seo_schema_type', $schema);
        }
    }

    /**
     * Import single term meta
     *
     * @param int $term_id Term ID
     * @return void
     */
    protected function import_single_term_meta(int $term_id): void {
        $term_meta_keys = [
            'rank_math_title' => '_prorank_seo_title',
            'rank_math_description' => '_prorank_seo_description',
            'rank_math_focus_keyword' => '_prorank_seo_focus_keyword',
            'rank_math_canonical_url' => '_prorank_seo_canonical',
            'rank_math_facebook_title' => '_prorank_seo_og_title',
            'rank_math_facebook_description' => '_prorank_seo_og_description',
            'rank_math_facebook_image' => '_prorank_seo_og_image',
        ];

        foreach ($term_meta_keys as $rm_key => $prorank_key) {
            $value = get_term_meta($term_id, $rm_key, true);

            if (!empty($value)) {
                update_term_meta($term_id, $prorank_key, $this->sanitize_meta_value($value));
            }
        }

        // Handle robots meta
        $robots = get_term_meta($term_id, 'rank_math_robots', true);
        if (is_array($robots)) {
            if (in_array('noindex', $robots, true)) {
                update_term_meta($term_id, '_prorank_seo_noindex', 'noindex');
            }
        }
    }

    /**
     * Import Rank Math settings
     *
     * @return void
     */
    protected function import_settings(): void {
        // Import general settings
        $general = get_option('rank-math-options-general');
        if ($general && is_array($general)) {
            $settings = [];

            // Title separator
            if (isset($general['title_separator'])) {
                $settings['title_separator'] = $general['title_separator'];
            }

            // Breadcrumbs settings
            if (isset($general['breadcrumbs'])) {
                $settings['enable_breadcrumbs'] = (bool) $general['breadcrumbs'];
            }

            if (!empty($settings)) {
                update_option('prorank_seo_settings', $settings);
                $this->stats['settings']++;
            }
        }

        // Import titles & meta settings
        $titles = get_option('rank-math-options-titles');
        if ($titles && is_array($titles)) {
            $title_settings = [];

            if (isset($titles['homepage_title'])) {
                $title_settings['home_title'] = $titles['homepage_title'];
            }
            if (isset($titles['homepage_description'])) {
                $title_settings['home_description'] = $titles['homepage_description'];
            }

            if (!empty($title_settings)) {
                $existing = get_option('prorank_seo_titles_meta', []);
                update_option('prorank_seo_titles_meta', array_merge($existing, $title_settings));
                $this->stats['settings']++;
            }
        }

        // Import social settings
        $social = get_option('rank-math-options-social');
        if ($social && is_array($social)) {
            $social_settings = [];

            if (isset($social['facebook_url'])) {
                $social_settings['facebook_page_url'] = $social['facebook_url'];
            }
            if (isset($social['twitter_author_names'])) {
                $social_settings['twitter_username'] = $social['twitter_author_names'];
            }

            if (!empty($social_settings)) {
                update_option('prorank_seo_social', $social_settings);
                $this->stats['settings']++;
            }
        }
    }

    /**
     * Import Rank Math redirects
     *
     * @return void
     */
    protected function import_redirects(): void {
        global $wpdb;

        // Rank Math stores redirects in rank_math_redirections table
        $table = $wpdb->prefix . 'rank_math_redirections';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table)) !== $table) {
            return;
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $redirects = $wpdb->get_results(
            "SELECT * FROM {$table} WHERE status = 'active'"
        );

        foreach ($redirects as $redirect) {
            $sources = maybe_unserialize($redirect->sources);
            $from_url = is_array($sources) && isset($sources[0]['pattern'])
                ? $sources[0]['pattern']
                : '';

            if (empty($from_url)) {
                continue;
            }

            $prorank_redirect = [
                'from' => sanitize_text_field($from_url),
                'to' => esc_url_raw($redirect->url_to),
                'type' => intval($redirect->header_code),
                'created' => current_time('mysql'),
            ];

            $this->save_redirect($prorank_redirect);
            $this->stats['redirects']++;
        }
    }

    /**
     * Import schema settings
     *
     * @return void
     */
    protected function import_schema(): void {
        $schema_options = get_option('rank-math-options-titles');

        if ($schema_options && is_array($schema_options)) {
            $prorank_schema = [];

            if (isset($schema_options['knowledgegraph_type'])) {
                $prorank_schema['site_representation'] = $schema_options['knowledgegraph_type'];
            }

            if (isset($schema_options['knowledgegraph_name'])) {
                $prorank_schema['organization_name'] = $schema_options['knowledgegraph_name'];
            }

            if (isset($schema_options['knowledgegraph_logo'])) {
                $prorank_schema['organization_logo'] = $schema_options['knowledgegraph_logo'];
            }

            if (!empty($prorank_schema)) {
                update_option('prorank_seo_schema', $prorank_schema);
                $this->stats['settings']++;
            }
        }
    }

    /**
     * Map meta key
     *
     * @param string $competitor_key Rank Math meta key
     * @return string ProRank meta key
     */
    protected function map_meta_key(string $competitor_key): string {
        return $this->meta_key_map[$competitor_key] ?? $competitor_key;
    }

    /**
     * Save redirect
     *
     * @param array $redirect Redirect data
     * @return void
     */
    private function save_redirect(array $redirect): void {
        $redirects = get_option('prorank_seo_redirects', []);
        $redirects[] = $redirect;
        update_option('prorank_seo_redirects', $redirects);
    }
}
