<?php
/**
 * Image Optimizer Interface
 *
 * @package ProRank\SEO\Core\ImageOptimization
 * @since   1.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core\ImageOptimization;

defined( 'ABSPATH' ) || exit;

/**
 * Interface for image optimizers
 */
interface OptimizerInterface {
	
	/**
	 * Check if this optimizer is available on the system
	 *
	 * @return bool
	 */
	public function is_available(): bool;
	
	/**
	 * Get optimizer priority (higher = preferred)
	 *
	 * @return int
	 */
	public function get_priority(): int;
	
	/**
	 * Get supported mime types
	 *
	 * @return array
	 */
	public function get_supported_types(): array;
	
	/**
	 * Optimize an image
	 *
	 * @param string $file_path Path to image file.
	 * @param array  $options   Optimization options.
	 * @return array{success: bool, original_size: int, optimized_size: int, method: string}
	 */
	public function optimize( string $file_path, array $options = [] ): array;
	
	/**
	 * Convert image to WebP
	 *
	 * @param string $file_path Path to image file.
	 * @param array  $options   Conversion options.
	 * @return array{success: bool, webp_path: string, original_size: int, webp_size: int}
	 */
	public function convert_to_webp( string $file_path, array $options = [] ): array;
	
	/**
	 * Convert image to AVIF
	 *
	 * @param string $file_path Path to image file.
	 * @param array  $options   Conversion options.
	 * @return array{success: bool, avif_path: string, original_size: int, avif_size: int}
	 */
	public function convert_to_avif( string $file_path, array $options = [] ): array;
	
	/**
	 * Get optimizer capabilities
	 *
	 * @return array
	 */
	public function get_capabilities(): array;
}