<?php
/**
 * Settings helper (static)
 *
 * @package ProRank\SEO\Core\Config
 */
declare(strict_types=1);

namespace ProRank\SEO\Core\Config;

defined( 'ABSPATH' ) || exit;

/**
 * Lightweight settings wrapper stored in the `prorank_settings` option.
 *
 * Supports:
 * - Simple keys: "onboarding_completed"
 * - Dot paths: "modules.sitemaps.news_enabled"
 */
final class Settings {
    private const OPTION_NAME = 'prorank_settings';

    /**
     * In-request cache for settings option.
     *
     * @var array<string, mixed>|null
     */
    private static ?array $cache = null;

    /**
     * Get a setting value.
     *
     * @param string $path Setting key or dot-path.
     * @param mixed  $default Default value if not found.
     * @return mixed
     */
    public static function get(string $path, $default = null) {
        $path = trim($path);
        if ($path === '') {
            return $default;
        }

        $settings = self::load();

        // Dot-path lookup (nested arrays).
        if (strpos($path, '.') !== false) {
            $value = self::getNested($settings, $path);
            if ($value !== null) {
                return $value;
            }

            // Fallbacks for legacy flat keys.
            if (array_key_exists($path, $settings)) {
                return $settings[$path];
            }
            $underscore = str_replace('.', '_', $path);
            if (array_key_exists($underscore, $settings)) {
                return $settings[$underscore];
            }

            return $default;
        }

        // Direct key.
        if (array_key_exists($path, $settings)) {
            return $settings[$path];
        }

        // Legacy group_* fallback (only when caller expects an array).
        if (is_array($default)) {
            $group = self::getGroupFromFlatKeys($settings, $path);
            if (!empty($group)) {
                return array_merge($default, $group);
            }
        }

        return $default;
    }

    /**
     * Set/update a setting value.
     *
     * @param string $path Setting key or dot-path.
     * @param mixed  $value Value to store.
     * @return bool
     */
    public static function set(string $path, $value): bool {
        $path = trim($path);
        if ($path === '') {
            return false;
        }

        $settings = self::load();

        if (strpos($path, '.') !== false) {
            self::setNested($settings, $path, $value);
            return self::save($settings);
        }

        $settings[$path] = $value;
        return self::save($settings);
    }

    /**
     * Alias for set() (legacy naming used by controllers).
     *
     * @param string $path
     * @param mixed  $value
     * @return bool
     */
    public static function update(string $path, $value): bool {
        return self::set($path, $value);
    }

    /**
     * Load settings from WP option.
     *
     * @return array<string, mixed>
     */
    private static function load(): array {
        if (self::$cache !== null) {
            return self::$cache;
        }

        $raw = get_option(self::OPTION_NAME, []);
        self::$cache = is_array($raw) ? $raw : [];
        return self::$cache;
    }

    /**
     * Persist settings to WP option (and refresh cache).
     *
     * @param array<string, mixed> $settings
     * @return bool
     */
    private static function save(array $settings): bool {
        self::$cache = $settings;
        return update_option(self::OPTION_NAME, $settings);
    }

    /**
     * Get nested value for a dot-path. Returns null if not found.
     *
     * @param array<string, mixed> $settings
     * @param string              $path
     * @return mixed|null
     */
    private static function getNested(array $settings, string $path) {
        $current = $settings;
        foreach (explode('.', $path) as $segment) {
            if ($segment === '' || !is_array($current) || !array_key_exists($segment, $current)) {
                return null;
            }
            $current = $current[$segment];
        }
        return $current;
    }

    /**
     * Set nested value for a dot-path.
     *
     * @param array<string, mixed> $settings
     * @param string              $path
     * @param mixed               $value
     * @return void
     */
    private static function setNested(array &$settings, string $path, $value): void {
        $segments = array_values(array_filter(explode('.', $path), static fn($s) => $s !== ''));
        if (empty($segments)) {
            return;
        }

        $last = array_pop($segments);
        $ref = &$settings;

        foreach ($segments as $segment) {
            if (!isset($ref[$segment]) || !is_array($ref[$segment])) {
                $ref[$segment] = [];
            }
            $ref = &$ref[$segment];
        }

        $ref[$last] = $value;
    }

    /**
     * Build a settings group array from legacy flat keys like "group_key".
     *
     * @param array<string, mixed> $settings
     * @param string              $group
     * @return array<string, mixed>
     */
    private static function getGroupFromFlatKeys(array $settings, string $group): array {
        $prefix = $group . '_';
        $prefixLength = strlen($prefix);
        $result = [];

        foreach ($settings as $key => $value) {
            if (!is_string($key) || strpos($key, $prefix) !== 0) {
                continue;
            }
            $subKey = substr($key, $prefixLength);
            if ($subKey === '') {
                continue;
            }
            $result[$subKey] = $value;
        }

        return $result;
    }
}

