<?php
/**
 * Centralized Output Buffer Handler
 *
 * Provides a single output buffer for all performance modules to hook into.
 * Based on WP Rocket's `rocket_buffer` and Perfmatters' buffer patterns.
 *
 * This ensures complete page coverage (headers, footers, sidebars) instead of
 * just content areas filtered via `the_content`.
 *
 * @package ProRank\SEO\Core
 * @since   2.0.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Buffer class for centralized output buffering
 */
class Buffer {

    /**
     * Whether buffer has been started
     *
     * @var bool
     */
    private static bool $started = false;

    /**
     * Request start time for performance tracking
     *
     * @var float
     */
    private static float $start_time;

    /**
     * Initialize the buffer system
     *
     * @return void
     */
    public static function init(): void {
        // Don't run in admin, AJAX, or cron contexts
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return;
        }

        // Don't run for REST API requests
        if (defined('REST_REQUEST') && REST_REQUEST) {
            return;
        }

        // Store request start time
        self::$start_time = microtime(true);

        // Capture output via template_include so the buffer opens and closes in one flow.
        add_filter('template_include', [__CLASS__, 'capture_template'], 9999);
    }

    /**
     * Capture template output for processing.
     *
     * @param string $template Template path.
     * @return string Template path to include after capture.
     * @return void
     */
    public static function capture_template(string $template): string {
        if (self::$started || !self::should_buffer()) {
            return $template;
        }

        self::$started = true;

        ob_start();
        include $template;
        $html = ob_get_clean();

        $html = self::process((string) $html);
        echo prorank_sanitize_full_html($html); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Sanitized via prorank_sanitize_full_html().

        return self::get_empty_template();
    }

    /**
     * Decide if buffering should run for this request.
     *
     * @return bool
     */
    private static function should_buffer(): bool {
        if (is_user_logged_in()) {
            $settings = get_option('prorank_performance_settings', []);
            if (empty($settings['optimize_logged_in'])) {
                return false;
            }
        }

        if (is_preview() || is_customize_preview()) {
            return false;
        }

        if (is_feed()) {
            return false;
        }

        if (is_404()) {
            $settings = get_option('prorank_performance_settings', []);
            if (empty($settings['optimize_404'])) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get a no-op template path to prevent double output.
     *
     * @return string
     */
    private static function get_empty_template(): string {
        return __DIR__ . '/empty-template.php';
    }

    /**
     * Process the buffered output
     *
     * This method applies all registered filters to the HTML output.
     * Filters should be added at specific priorities:
     *   - 5:  Early processing (e.g., remove comments)
     *   - 10: Critical CSS injection
     *   - 15: Font optimization (preload, font-display)
     *   - 20: JavaScript optimization (defer, delay)
     *   - 25: Lazy loading (images, iframes, videos)
     *   - 30: CLS prevention (add dimensions)
     *   - 35: HTML minification
     *   - 40: Final cleanup
     *
     * @param string $html The buffered HTML output
     * @return string Processed HTML
     */
    public static function process(string $html): string {
        // Quick validation - only process full HTML pages
        if (!self::is_valid_html($html)) {
            return $html;
        }

        // Allow other plugins to check if buffer is active
        if (!apply_filters('prorank_buffer_enabled', true)) {
            return $html;
        }

        /**
         * Main filter for HTML processing
         *
         * Performance modules should hook into this filter:
         *
         * @example
         * add_filter('prorank_output_buffer', [$this, 'process_html'], 25);
         *
         * @param string $html The full page HTML
         * @return string Modified HTML
         */
        $html = apply_filters('prorank_output_buffer', $html);

        // Add debug comment if enabled
        if (self::is_debug_enabled()) {
            $html = self::add_debug_comment($html);
        }

        return $html;
    }

    /**
     * Check if HTML is valid for processing
     *
     * @param string $html HTML content
     * @return bool
     */
    private static function is_valid_html(string $html): bool {
        // Skip empty content
        if (empty($html)) {
            return false;
        }

        // Must have HTML structure
        if (stripos($html, '<html') === false) {
            return false;
        }

        // Must have body closing tag (indicates full page)
        if (stripos($html, '</body>') === false) {
            return false;
        }

        // Skip if content looks like JSON
        if (trim($html)[0] === '{' || trim($html)[0] === '[') {
            return false;
        }

        // Skip if content looks like XML
        if (stripos($html, '<?xml') === 0) {
            return false;
        }

        return true;
    }

    /**
     * Check if debug mode is enabled
     *
     * @return bool
     */
    private static function is_debug_enabled(): bool {
        // Check for constant
        if (defined('PRORANK_DEBUG') && PRORANK_DEBUG) {
            return true;
        }

        // Check for setting
        $settings = get_option('prorank_advanced_settings', []);
        return !empty($settings['debug_mode']);
    }

    /**
     * Add debug comment to HTML
     *
     * @param string $html HTML content
     * @return string HTML with debug comment
     */
    private static function add_debug_comment(string $html): string {
        $elapsed = round((microtime(true) - self::$start_time) * 1000, 2);
        $memory = round(memory_get_peak_usage() / 1024 / 1024, 2);

        $comment = sprintf(
            "\n<!-- ProRank SEO | Buffer processed in %sms | Memory: %sMB -->\n",
            $elapsed,
            $memory
        );

        // Insert before </body>
        $pos = stripos($html, '</body>');
        if ($pos !== false) {
            $html = substr_replace($html, $comment, $pos, 0);
        }

        return $html;
    }

    /**
     * Check if buffer has been started
     *
     * @return bool
     */
    public static function is_started(): bool {
        return self::$started;
    }

    /**
     * Get elapsed time since buffer start
     *
     * @return float Time in milliseconds
     */
    public static function get_elapsed_time(): float {
        if (!isset(self::$start_time)) {
            return 0.0;
        }
        return (microtime(true) - self::$start_time) * 1000;
    }
}
