<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Uses custom tables with safe prepared queries
/**
 * Basic (Local) Audit Engine
 *
 * Lightweight audit that runs locally without a license/server calls.
 *
 * @package ProRank\SEO\Core\Audits
 */
declare(strict_types=1);

namespace ProRank\SEO\Core\Audits;

defined( 'ABSPATH' ) || exit;

use WP_Error;

class BasicAuditEngine {
    private const AUDIT_ID_PREFIX = 'local_basic_';

    private string $audits_table;
    private string $audit_urls_table;
    private string $audit_issues_table;
    private array $column_cache = [];

    public function __construct() {
        global $wpdb;
        $this->audits_table = $wpdb->prefix . 'prorank_audits';
        $this->audit_urls_table = $wpdb->prefix . 'prorank_audit_urls';
        $this->audit_issues_table = $wpdb->prefix . 'prorank_audit_issues';
    }

    /**
     * Run a basic audit synchronously and persist results.
     *
     * @param array $settings Optional settings (unused for now).
     * @return array|WP_Error Status payload on success, WP_Error on failure.
     */
    public function run(array $settings = []) {
        $start = $this->start_audit($settings);
        if (is_wp_error($start)) {
            return $start;
        }

        return $this->process_audit((string) $start['audit_id']);
    }

    /**
     * Start an audit and record initial status.
     *
     * @param array $settings Audit settings
     * @return array|WP_Error
     */
    public function start_audit(array $settings = []) {
        global $wpdb;

        $this->maybe_create_tables();

        // Prevent overlapping audits.
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $running = $wpdb->get_var($wpdb->prepare(
            "SELECT audit_id FROM {$this->audits_table} WHERE status IN (%s, %s, %s) ORDER BY started_at DESC, id DESC LIMIT 1",
            'crawling',
            'checking',
            'running'
        ));
        if ($running) {
            return new WP_Error('audit_running', __('An audit is already running.', 'prorank-seo'));
        }

        $audit_id = self::AUDIT_ID_PREFIX . wp_generate_uuid4();
        $now_mysql = current_time('mysql');

        // Get max URLs from settings - free version allows up to 100 pages
        $max_urls = 50; // Default for free version
        if (isset($settings['max_urls']) && is_numeric($settings['max_urls'])) {
            $max_urls = min(max((int) $settings['max_urls'], 1), 100);
        }

        $max_depth = 5;
        if (isset($settings['max_depth']) && is_numeric($settings['max_depth'])) {
            $max_depth = min(max((int) $settings['max_depth'], 1), 20);
        }

        // Collect URLs to audit
        $urls_to_check = $this->get_urls_to_audit($max_urls, $max_depth);
        $total_urls = count($urls_to_check);

        set_transient('prorank_audit_urls_' . $audit_id, $urls_to_check, HOUR_IN_SECONDS);

        $progress = $this->build_progress(0, $total_urls);

        $data = [
            'audit_id' => $audit_id,
            'status' => 'checking',
            'started_at' => $now_mysql,
            'total_urls' => $total_urls,
            'options' => wp_json_encode($settings),
        ];
        $format = ['%s', '%s', '%s', '%d', '%s'];

        if ($this->has_column($this->audits_table, 'progress')) {
            $data['progress'] = wp_json_encode($progress);
            $format[] = '%s';
        }
        if ($this->has_column($this->audits_table, 'pages_crawled')) {
            $data['pages_crawled'] = 0;
            $format[] = '%d';
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $inserted = $wpdb->insert($this->audits_table, $data, $format);
        if ($inserted === false) {
            return new WP_Error('db_error', __('Failed to start audit', 'prorank-seo'));
        }

        update_option('prorank_audit_current_id', $audit_id, false);
        $this->update_progress($audit_id, 0, $total_urls);

        return [
            'audit_id' => $audit_id,
            'total_urls' => $total_urls,
            'progress' => $progress,
        ];
    }

    /**
     * Process a started audit.
     *
     * @param string $audit_id Audit ID
     * @return array|WP_Error
     */
    public function process_audit(string $audit_id) {
        global $wpdb;

        $this->maybe_create_tables();

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $audit = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->audits_table} WHERE audit_id = %s LIMIT 1",
            $audit_id
        ));
        if (!$audit) {
            return new WP_Error('audit_not_found', __('Audit not found.', 'prorank-seo'));
        }

        $settings = [];
        if (!empty($audit->options)) {
            $decoded = json_decode((string) $audit->options, true);
            if (is_array($decoded)) {
                $settings = $decoded;
            }
        }

        $max_urls = 50; // Default for free version
        if (isset($settings['max_urls']) && is_numeric($settings['max_urls'])) {
            $max_urls = min(max((int) $settings['max_urls'], 1), 100);
        }

        $max_depth = 5;
        if (isset($settings['max_depth']) && is_numeric($settings['max_depth'])) {
            $max_depth = min(max((int) $settings['max_depth'], 1), 20);
        }

        $urls_to_check = get_transient('prorank_audit_urls_' . $audit_id);
        if (!is_array($urls_to_check) || empty($urls_to_check)) {
            $urls_to_check = $this->get_urls_to_audit($max_urls, $max_depth);
        }
        $total_urls = count($urls_to_check);

        // Refresh status + totals for the running audit.
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update(
            $this->audits_table,
            [
                'status' => 'checking',
                'total_urls' => $total_urls,
            ],
            ['audit_id' => $audit_id],
            ['%s', '%d'],
            ['%s']
        );

        $this->update_progress($audit_id, 0, $total_urls);

        $now_mysql = !empty($audit->started_at) ? (string) $audit->started_at : current_time('mysql');

        // Collect site-wide issues (settings, sitemap, robots.txt)
        $site_issues = $this->collect_site_issues($settings);

        // Audit each page
        $all_issues = $site_issues;
        $url_results = [];

        $checked = 0;
        foreach ($urls_to_check as $url_info) {
            $url = $url_info['url'];
            $page_issues = $this->check_page_content($url, $url_info['type'], $settings);

            foreach ($page_issues as &$issue) {
                $issue['url'] = $url;
            }

            $all_issues = array_merge($all_issues, $page_issues);
            $url_results[] = [
                'url' => $url,
                'type' => $url_info['type'],
                'issues' => $page_issues,
            ];

            $checked++;
            $this->update_progress($audit_id, $checked, $total_urls);
        }

        $counts = $this->count_severities($all_issues);
        $score = $this->calculate_score($counts);

        $stats = [
            'mode' => 'local_basic',
            'unique_issues' => [
                'critical' => $counts['critical'],
                'high' => $counts['high'],
                'medium' => $counts['medium'],
                'low' => $counts['low'],
                'warning' => $counts['warning'],
                'total' => $counts['total'],
            ],
        ];

        $completed_at = current_time('mysql');
        $progress = $this->build_progress($total_urls, $total_urls);

        $update_data = [
            'status' => 'completed',
            'completed_at' => $completed_at,
            'total_urls' => $total_urls,
            'score' => $score,
            'stats' => wp_json_encode($stats),
        ];
        $update_format = ['%s', '%s', '%d', '%d', '%s'];

        if ($this->has_column($this->audits_table, 'progress')) {
            $update_data['progress'] = wp_json_encode($progress);
            $update_format[] = '%s';
        }
        if ($this->has_column($this->audits_table, 'pages_crawled')) {
            $update_data['pages_crawled'] = $total_urls;
            $update_format[] = '%d';
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->update(
            $this->audits_table,
            $update_data,
            ['audit_id' => $audit_id],
            $update_format,
            ['%s']
        );

        // Store URL entries and issues
        foreach ($url_results as $result) {
            $url_issues = $result['issues'];
            $url_counts = $this->count_severities($url_issues);
            $url_score = $this->calculate_score($url_counts);

            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->insert(
                $this->audit_urls_table,
                [
                    'audit_id' => $audit_id,
                    'url' => $result['url'],
                    'status' => 'checked',
                    'issues_count' => $url_counts['critical'] + $url_counts['high'],
                    'warnings_count' => $url_counts['medium'] + $url_counts['low'],
                    'passed_count' => 0,
                    'score' => $url_score,
                    'created_at' => $now_mysql,
                    'checked_at' => $completed_at,
                ],
                ['%s', '%s', '%s', '%d', '%d', '%d', '%d', '%s', '%s']
            );

            $url_id = (int) $wpdb->insert_id;

            foreach ($url_issues as $issue) {
                // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                $wpdb->insert(
                    $this->audit_issues_table,
                    [
                        'audit_id' => $audit_id,
                        'url_id' => $url_id,
                        'type' => $issue['type'],
                        'severity' => $issue['severity'],
                        'message' => $issue['message'],
                        'data' => wp_json_encode($issue),
                        'created_at' => $now_mysql,
                    ],
                    ['%s', '%d', '%s', '%s', '%s', '%s', '%s']
                );
            }
        }

        // Also store site-wide issues (attach to first URL)
        if (!empty($site_issues)) {
            $first_url_id = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM {$this->audit_urls_table} WHERE audit_id = %s LIMIT 1",
                $audit_id
            ));

            if ($first_url_id) {
                foreach ($site_issues as $issue) {
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $wpdb->insert(
                        $this->audit_issues_table,
                        [
                            'audit_id' => $audit_id,
                            'url_id' => (int) $first_url_id,
                            'type' => $issue['type'],
                            'severity' => $issue['severity'],
                            'message' => $issue['message'],
                            'data' => wp_json_encode($issue),
                            'created_at' => $now_mysql,
                        ],
                        ['%s', '%d', '%s', '%s', '%s', '%s', '%s']
                    );
                }
            }
        }

        $this->update_progress($audit_id, $total_urls, $total_urls);
        delete_transient('prorank_audit_urls_' . $audit_id);
        delete_transient('prorank_audit_progress_' . $audit_id);
        delete_transient('prorank_dashboard_stats');
        update_option('prorank_audit_current_id', '', false);

        return $this->format_status($audit_id, [
            'started_at' => $now_mysql,
            'completed_at' => $completed_at,
            'total_urls' => $total_urls,
            'score' => $score,
        ], $counts);
    }

    /**
     * Get URLs to audit (homepage + posts + pages + products + custom post types)
     */
    private function get_urls_to_audit(int $max_urls = 50, int $max_depth = 0): array {
        $urls = [];
        $added_urls = []; // Track URLs to avoid duplicates

        // Always include homepage
        $home = home_url('/');
        $home_path = (string) wp_parse_url($home, PHP_URL_PATH);
        if ($home_path === '') {
            $home_path = '/';
        }
        $urls[] = [
            'url' => $home,
            'type' => 'homepage',
        ];
        $added_urls[$home] = true;

        // Calculate remaining slots
        $remaining = $max_urls - 1;
        if ($remaining <= 0) {
            return $urls;
        }

        // Get all public post types
        $post_types = get_post_types(['public' => true], 'names');
        // Prioritize: posts, pages, products, then others
        $priority_types = ['post', 'page', 'product'];
        $other_types = array_diff($post_types, $priority_types, ['attachment']);

        // Allocate slots: 40% posts, 30% pages, 20% products, 10% other
        $slots = [
            'post' => (int) ceil($remaining * 0.4),
            'page' => (int) ceil($remaining * 0.3),
            'product' => (int) ceil($remaining * 0.2),
            'other' => (int) ceil($remaining * 0.1),
        ];

        // Get posts
        if (in_array('post', $post_types, true)) {
            $posts = get_posts([
                'numberposts' => $slots['post'],
                'post_type' => 'post',
                'post_status' => 'publish',
                'orderby' => 'date',
                'order' => 'DESC',
            ]);

            foreach ($posts as $post) {
                $permalink = get_permalink($post);
                if ($this->is_url_within_depth($permalink, $max_depth, $home_path) && !isset($added_urls[$permalink])) {
                    $urls[] = ['url' => $permalink, 'type' => 'post'];
                    $added_urls[$permalink] = true;
                }
            }
        }

        // Get pages
        if (in_array('page', $post_types, true)) {
            $pages = get_posts([
                'numberposts' => $slots['page'],
                'post_type' => 'page',
                'post_status' => 'publish',
                'orderby' => 'modified',
                'order' => 'DESC',
            ]);

            foreach ($pages as $page) {
                $permalink = get_permalink($page);
                // Skip homepage
                if ($this->is_url_within_depth($permalink, $max_depth, $home_path) && !isset($added_urls[$permalink]) && $permalink !== $home) {
                    $urls[] = ['url' => $permalink, 'type' => 'page'];
                    $added_urls[$permalink] = true;
                }
            }
        }

        // Get WooCommerce products if available
        if (in_array('product', $post_types, true)) {
            $products = get_posts([
                'numberposts' => $slots['product'],
                'post_type' => 'product',
                'post_status' => 'publish',
                'orderby' => 'date',
                'order' => 'DESC',
            ]);

            foreach ($products as $product) {
                $permalink = get_permalink($product);
                if ($this->is_url_within_depth($permalink, $max_depth, $home_path) && !isset($added_urls[$permalink])) {
                    $urls[] = ['url' => $permalink, 'type' => 'product'];
                    $added_urls[$permalink] = true;
                }
            }
        }

        // Get other custom post types
        if (!empty($other_types) && $slots['other'] > 0) {
            $per_type = max(1, (int) floor($slots['other'] / count($other_types)));
            foreach ($other_types as $post_type) {
                $items = get_posts([
                    'numberposts' => $per_type,
                    'post_type' => $post_type,
                    'post_status' => 'publish',
                    'orderby' => 'date',
                    'order' => 'DESC',
                ]);

                foreach ($items as $item) {
                    $permalink = get_permalink($item);
                    if ($this->is_url_within_depth($permalink, $max_depth, $home_path) && !isset($added_urls[$permalink])) {
                        $urls[] = ['url' => $permalink, 'type' => $post_type];
                        $added_urls[$permalink] = true;
                    }
                }
            }
        }

        // Log for debugging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            prorank_log('[ProRank Audit] Requested max_urls: ' . $max_urls . ', max_depth: ' . $max_depth . ', Found URLs: ' . count($urls));
        }

        return array_slice($urls, 0, $max_urls);
    }

    /**
     * Check if URL is within max depth (relative to the site root).
     */
    private function is_url_within_depth(string $url, int $max_depth, string $home_path): bool {
        if ($max_depth <= 0) {
            return true;
        }

        $path = (string) wp_parse_url($url, PHP_URL_PATH);
        if ($path === '') {
            return true;
        }

        if ($home_path !== '/' && str_starts_with($path, $home_path)) {
            $path = substr($path, strlen($home_path));
        }

        $path = trim($path, '/');
        if ($path === '') {
            return true;
        }

        $depth = count(array_filter(explode('/', $path), 'strlen'));

        return $depth <= $max_depth;
    }

    /**
     * Collect site-wide issues (not page-specific)
     *
     * @param array $settings Audit settings.
     */
    private function collect_site_issues(array $settings = []): array {
        $issues = [];

        $site_url = home_url('/');
        $using_https = function_exists('wp_is_using_https')
            ? (bool) wp_is_using_https()
            : (stripos($site_url, 'https://') === 0);

        // Technical SEO: Search engine visibility
        $blog_public = (int) get_option('blog_public', 1);
        if ($blog_public !== 1) {
            $issues[] = $this->issue([
                'type' => 'discourage_search_engines',
                'severity' => 'critical',
                'category' => 'technical_seo',
                'title' => __('Search engine visibility is disabled', 'prorank-seo'),
                'description' => __('Your site is set to discourage search engines from indexing it.', 'prorank-seo'),
                'how_to_fix' => __('Go to Settings → Reading and uncheck "Discourage search engines from indexing this site".', 'prorank-seo'),
            ]);
        }

        // Technical SEO: Permalinks
        $permalink_structure = (string) get_option('permalink_structure', '');
        if (trim($permalink_structure) === '') {
            $issues[] = $this->issue([
                'type' => 'permalink_structure',
                'severity' => 'medium',
                'category' => 'technical_seo',
                'title' => __('Permalinks are not enabled', 'prorank-seo'),
                'description' => __('Pretty permalinks improve SEO and readability.', 'prorank-seo'),
                'how_to_fix' => __('Go to Settings → Permalinks and select "Post name".', 'prorank-seo'),
            ]);
        }

        // Security: HTTPS
        if ($this->is_check_enabled($settings, 'https_status', true) && !$using_https) {
            $issues[] = $this->issue([
                'type' => 'https_not_enabled',
                'severity' => 'high',
                'category' => 'security',
                'title' => __('HTTPS is not enabled', 'prorank-seo'),
                'description' => __('Your site is not using HTTPS, which can hurt SEO and security.', 'prorank-seo'),
                'how_to_fix' => __('Install an SSL certificate and update WordPress Address + Site Address to https://', 'prorank-seo'),
            ]);
        }

        // Site basics: Site icon
        if (function_exists('has_site_icon') && !has_site_icon()) {
            $issues[] = $this->issue([
                'type' => 'site_icon_missing',
                'severity' => 'low',
                'category' => 'on_page_seo',
                'title' => __('Site icon (favicon) is missing', 'prorank-seo'),
                'description' => __('A site icon improves branding in tabs, bookmarks, and some SERP displays.', 'prorank-seo'),
                'how_to_fix' => __('Go to Appearance → Customize → Site Identity and set a Site Icon.', 'prorank-seo'),
            ]);
        }

        // Site basics: Default tagline
        $tagline = trim((string) get_option('blogdescription', ''));
        if ($tagline !== '' && strtolower($tagline) === 'just another wordpress site') {
            $issues[] = $this->issue([
                'type' => 'default_tagline',
                'severity' => 'low',
                'category' => 'on_page_seo',
                'title' => __('Default WordPress tagline detected', 'prorank-seo'),
                'description' => __('Your site tagline is still the default "Just another WordPress site".', 'prorank-seo'),
                'how_to_fix' => __('Go to Settings → General and update the Tagline to match your brand.', 'prorank-seo'),
            ]);
        }

        // Sitemap & robots.txt checks
        if ($this->is_check_enabled($settings, 'xml_sitemap', true)) {
            $issues = array_merge($issues, $this->check_sitemap());
        }
        if ($this->is_check_enabled($settings, 'robots_txt', true)) {
            $issues = array_merge($issues, $this->check_robots_txt());
        }

        return $issues;
    }

    /**
     * Return latest basic audit status (if any).
     */
    public function get_latest_status(): ?array {
        global $wpdb;

        if (!$this->table_exists($this->audits_table)) {
            return null;
        }

        $select = 'audit_id, status, started_at, completed_at, total_urls, score';
        if ($this->has_column($this->audits_table, 'progress')) {
            $select .= ', progress';
        }
        if ($this->has_column($this->audits_table, 'pages_crawled')) {
            $select .= ', pages_crawled';
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $row = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT {$select}
                 FROM {$this->audits_table}
                 WHERE status IN (%s, %s, %s)
                 ORDER BY started_at DESC, id DESC
                 LIMIT 1",
                'crawling',
                'checking',
                'running'
            )
        );

        if (!$row) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $row = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT {$select}
                     FROM {$this->audits_table}
                     WHERE status = %s
                     ORDER BY completed_at DESC, started_at DESC, id DESC
                     LIMIT 1",
                    'completed'
                )
            );
        }

        if (!$row) {
            return null;
        }

        $counts = $this->get_issue_counts((string) $row->audit_id);

        if ($row->status !== 'completed') {
            $progress = $this->get_progress($row->audit_id, $row->progress ?? null, (int) ($row->total_urls ?? 0));

            return [
                'audit_id' => (string) $row->audit_id,
                'state' => (string) ($row->status ?: 'checking'),
                'status' => (string) ($row->status ?: 'checking'),
                'mode' => 'local_basic',
                'message' => __('Audit in progress', 'prorank-seo'),
                'progress' => $progress,
                'started_at' => $row->started_at,
                'completed_at' => $row->completed_at,
                'issuesCritical' => $counts['critical'],
                'issuesHigh' => $counts['high'],
                'issuesMedium' => $counts['medium'],
                'issuesLow' => $counts['low'],
                'issuesWarning' => $counts['warning'],
                'overallScore' => (int) ($row->score ?? 0),
                'overall_score' => (int) ($row->score ?? 0),
                'totalUrls' => (int) ($row->total_urls ?? 0),
                'total_urls' => (int) ($row->total_urls ?? 0),
            ];
        }

        return $this->format_status((string) $row->audit_id, [
            'started_at' => $row->started_at,
            'completed_at' => $row->completed_at,
            'total_urls' => (int) ($row->total_urls ?? 0),
            'score' => (int) ($row->score ?? 0),
        ], $counts);
    }

    /**
     * Get issues for a specific local audit.
     */
    public function get_issues(string $audit_id): array {
        global $wpdb;

        if (
            !$this->table_exists($this->audit_issues_table) ||
            !$this->table_exists($this->audit_urls_table)
        ) {
            return [];
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $rows = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT i.id, i.type, i.severity, i.message, i.data, u.url
                 FROM {$this->audit_issues_table} i
                 LEFT JOIN {$this->audit_urls_table} u ON u.id = i.url_id
                 WHERE i.audit_id = %s
                 ORDER BY FIELD(i.severity, 'critical', 'high', 'medium', 'low', 'passed'), i.id ASC",
                $audit_id
            )
        );

        if (empty($rows)) {
            return [];
        }

        $issues = [];
        foreach ($rows as $row) {
            $data = [];
            if (!empty($row->data)) {
                $decoded = json_decode((string) $row->data, true);
                if (is_array($decoded)) {
                    $data = $decoded;
                }
            }

            $issues[] = [
                'id' => (string) ($data['id'] ?? $row->id),
                'type' => (string) ($data['type'] ?? $row->type),
                'issue_type' => (string) ($data['type'] ?? $row->type),
                'severity' => (string) ($data['severity'] ?? $row->severity),
                'category' => (string) ($data['category'] ?? 'content'),
                'display_category' => (string) ($data['display_category'] ?? ($data['category'] ?? 'content')),
                'title' => (string) ($data['title'] ?? $row->type),
                'description' => (string) ($data['description'] ?? $row->message),
                'url' => (string) ($data['url'] ?? ($row->url ?? home_url('/'))),
                'impact' => (string) ($data['impact'] ?? ''),
                'how_to_fix' => (string) ($data['how_to_fix'] ?? ''),
                'fix_suggestion' => (string) ($data['how_to_fix'] ?? ''),
                'reference' => $data['reference'] ?? null,
            ];
        }

        return $issues;
    }

    /**
     * Get basic audit history.
     */
    public function get_history(int $limit = 10): array {
        global $wpdb;

        if (!$this->table_exists($this->audits_table)) {
            return [];
        }

        $limit = max(1, min(50, $limit));

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $rows = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT audit_id, status, started_at, completed_at, total_urls, score
                 FROM {$this->audits_table}
                 WHERE audit_id LIKE %s
                 ORDER BY completed_at DESC, started_at DESC, id DESC
                 LIMIT %d",
                self::AUDIT_ID_PREFIX . '%',
                $limit
            )
        );

        if (empty($rows)) {
            return [];
        }

        $domain = wp_parse_url(home_url('/'), PHP_URL_HOST) ?: '';

        $audits = [];
        foreach ($rows as $row) {
            $audit_id = (string) $row->audit_id;
            $counts = $this->get_issue_counts($audit_id);

            $audits[] = [
                'id' => $audit_id,
                'audit_id' => $audit_id,
                'domain' => $domain,
                'status' => (string) ($row->status ?? 'completed'),
                'state' => 'completed',
                'audit_type' => 'local_basic',
                'total_urls' => (int) ($row->total_urls ?? 0),
                'progress' => 100,
                'overallScore' => (int) ($row->score ?? 0),
                'overall_score' => (int) ($row->score ?? 0),
                'issues' => [
                    'critical' => $counts['critical'],
                    'high' => $counts['high'],
                    'medium' => $counts['medium'],
                    'low' => $counts['low'],
                    'warning' => $counts['warning'],
                    'total' => $counts['total'],
                ],
                'started_at' => $row->started_at,
                'completed_at' => $row->completed_at,
                'created_at' => $row->completed_at ?: $row->started_at,
            ];
        }

        return $audits;
    }

    private function format_status(string $audit_id, array $audit_row, array $counts): array {
        $total_urls = (int) ($audit_row['total_urls'] ?? 0);
        if ($total_urls <= 0) {
            $total_urls = 1;
        }

        return [
            'audit_id' => $audit_id,
            'state' => 'completed',
            'status' => 'completed',
            'mode' => 'local_basic',
            'message' => __('Basic audit completed locally', 'prorank-seo'),
            'progress' => [
                'percent' => 100,
                'percentage' => 100,
                'total_urls' => $total_urls,
                'checked_urls' => $total_urls,
            ],
            'started_at' => $audit_row['started_at'] ?? null,
            'completed_at' => $audit_row['completed_at'] ?? null,
            'issuesCritical' => $counts['critical'],
            'issuesHigh' => $counts['high'],
            'issuesMedium' => $counts['medium'],
            'issuesLow' => $counts['low'],
            'issuesWarning' => $counts['warning'],
            'overallScore' => (int) ($audit_row['score'] ?? 0),
            'overall_score' => (int) ($audit_row['score'] ?? 0),
            'totalUrls' => $total_urls,
            'total_urls' => $total_urls,
        ];
    }

    /**
     * Build progress payload for the audit UI.
     */
    private function build_progress(int $checked, int $total): array {
        $total = max(0, $total);
        $checked = max(0, min($checked, $total));
        $percent = $total > 0 ? (int) min(100, round(($checked / $total) * 100)) : 0;

        return [
            'percent' => $percent,
            'percentage' => $percent,
            'checked_urls' => $checked,
            'total_urls' => $total,
        ];
    }

    /**
     * Persist audit progress for polling endpoints.
     */
    private function update_progress(string $audit_id, int $checked, int $total): void {
        global $wpdb;

        $progress = $this->build_progress($checked, $total);

        update_option('prorank_audit_last_progress', [
            'checked' => $checked,
            'total' => $total,
            'percent' => $progress['percent'],
            'time' => time(),
        ], false);

        set_transient('prorank_audit_progress_' . $audit_id, $progress, HOUR_IN_SECONDS);

        $update = [];
        $format = [];

        if ($this->has_column($this->audits_table, 'progress')) {
            $update['progress'] = wp_json_encode($progress);
            $format[] = '%s';
        }
        if ($this->has_column($this->audits_table, 'pages_crawled')) {
            $update['pages_crawled'] = $checked;
            $format[] = '%d';
        }

        if (!empty($update)) {
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->update(
                $this->audits_table,
                $update,
                ['audit_id' => $audit_id],
                $format,
                ['%s']
            );
        }
    }

    /**
     * Resolve progress payload from DB/option cache.
     */
    private function get_progress(string $audit_id, $raw_progress, int $total_urls): array {
        $progress = [];
        if (is_string($raw_progress) && $raw_progress !== '') {
            $decoded = json_decode($raw_progress, true);
            if (is_array($decoded)) {
                $progress = $decoded;
            }
        }

        if (empty($progress)) {
            $cached = get_transient('prorank_audit_progress_' . $audit_id);
            if (is_array($cached)) {
                $progress = $cached;
            }
        }

        if (empty($progress)) {
            $last = get_option('prorank_audit_last_progress', []);
            if (is_array($last)) {
                $progress = [
                    'checked_urls' => (int) ($last['checked'] ?? 0),
                    'total_urls' => (int) ($last['total'] ?? $total_urls),
                    'percent' => (int) ($last['percent'] ?? 0),
                    'percentage' => (int) ($last['percent'] ?? 0),
                ];
            }
        }

        if (empty($progress)) {
            $progress = $this->build_progress(0, $total_urls);
        }

        return $progress;
    }

    /**
     * Check page content for SEO issues
     *
     * @param array $settings Audit settings.
     */
    private function check_page_content(string $url, string $page_type = 'page', array $settings = []): array {
        $issues = [];
        $settings = is_array($settings) ? $settings : [];

        // Fetch page
        $response = wp_remote_get($url, [
            'timeout' => 10,
            'sslverify' => false,
            'user-agent' => 'ProRank SEO Audit Bot/1.0',
        ]);

        if (is_wp_error($response)) {
            $issues[] = $this->issue([
                'type' => 'page_unreachable',
                'severity' => 'critical',
                'category' => 'technical_seo',
                'title' => sprintf(
                    /* translators: %s: placeholder value */
                    __('%s is unreachable', 'prorank-seo'), ucfirst($page_type)),
                'description' => sprintf(
                    /* translators: %s: error message */
                    __('Failed to fetch page: %s', 'prorank-seo'), $response->get_error_message()),
                'how_to_fix' => __('Check your server configuration and ensure the page is accessible.', 'prorank-seo'),
                'url' => $url,
            ]);
            return $issues;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        // Check HTTP status
        if ($status_code >= 400) {
            $issues[] = $this->issue([
                'type' => 'page_error',
                'severity' => 'critical',
                'category' => 'technical_seo',
                'title' => sprintf(
                    /* translators: %s: numeric value */
                    __('Page returns HTTP %d error', 'prorank-seo'), $status_code),
                'description' => sprintf(
                    /* translators: %s: error message */
                    __('This %s is returning an error status code.', 'prorank-seo'), $page_type),
                'how_to_fix' => __('Check your server logs and fix the error causing the page to fail.', 'prorank-seo'),
                'url' => $url,
            ]);
            return $issues;
        }

        // Parse HTML
        if ( ! class_exists( '\DOMDocument' ) || ! class_exists( '\DOMXPath' ) ) {
            $issues[] = $this->issue([
                'type' => 'missing_dom_extension',
                'severity' => 'critical',
                'category' => 'technical_seo',
                'title' => __('Server is missing the DOM extension', 'prorank-seo'),
                'description' => __('The PHP DOM extension is required to analyze page content.', 'prorank-seo'),
                'how_to_fix' => __('Ask your host to enable the PHP DOM extension.', 'prorank-seo'),
                'url' => $url,
            ]);
            return $issues;
        }

        $dom = new \DOMDocument();
        libxml_use_internal_errors(true);
        $body_for_dom = wp_check_invalid_utf8( $body );
        $body_for_dom = '<meta http-equiv="Content-Type" content="text/html; charset=utf-8">' . $body_for_dom;
        @$dom->loadHTML($body_for_dom);
        libxml_clear_errors();
        $xpath = new \DOMXPath($dom);

        $page_label = ucfirst($page_type);

        if ($this->is_check_enabled($settings, 'meta_tags', true)) {
            // Check title tag
            $titles = $xpath->query('//title');
            if ($titles->length === 0) {
                $issues[] = $this->issue([
                    'type' => 'missing_title',
                    'severity' => 'critical',
                    'category' => 'on_page_seo',
                    'title' => sprintf(
                        /* translators: %s: placeholder value */
                        __('%s is missing a title tag', 'prorank-seo'), $page_label),
                    'description' => __('The title tag is essential for SEO and click-through rates.', 'prorank-seo'),
                    'how_to_fix' => __('Add a title tag or configure it in ProRank → On-Page SEO → Titles & Meta.', 'prorank-seo'),
                    'url' => $url,
                ]);
            } else {
                $title_text = trim($titles->item(0)->textContent);
                if (strlen($title_text) < 10) {
                    $issues[] = $this->issue([
                        'type' => 'short_title',
                        'severity' => 'medium',
                        'category' => 'on_page_seo',
                        'title' => sprintf(
                            /* translators: %s: placeholder value */
                            __('%s title is too short', 'prorank-seo'), $page_label),
                        'description' => sprintf(/* translators: 1: title text 2: character count */
                            __('Title "%1$s" is only %2$d characters. Aim for 50-60.', 'prorank-seo'), esc_html(substr($title_text, 0, 30)), strlen($title_text)),
                        'how_to_fix' => __('Write a more descriptive title that includes your main keywords.', 'prorank-seo'),
                        'url' => $url,
                    ]);
                } elseif (strlen($title_text) > 70) {
                    $issues[] = $this->issue([
                        'type' => 'long_title',
                        'severity' => 'low',
                        'category' => 'on_page_seo',
                        'title' => sprintf(
                            /* translators: %s: placeholder value */
                            __('%s title may be truncated', 'prorank-seo'), $page_label),
                        'description' => sprintf(
                            /* translators: %s: numeric value */
                            __('Title is %d characters. Google typically displays ~60 characters.', 'prorank-seo'), strlen($title_text)),
                        'how_to_fix' => __('Consider shortening the title to under 60 characters.', 'prorank-seo'),
                        'url' => $url,
                    ]);
                }
            }
        }

        if ($this->is_check_enabled($settings, 'meta_tags', true)) {
            // Check meta description
            $meta_desc = $xpath->query('//meta[@name="description"]/@content');
            if ($meta_desc->length === 0) {
                $issues[] = $this->issue([
                    'type' => 'missing_meta_description',
                    'severity' => 'high',
                    'category' => 'on_page_seo',
                    'title' => sprintf(
                        /* translators: %s: placeholder value */
                        __('%s is missing a meta description', 'prorank-seo'), $page_label),
                    'description' => __('Meta descriptions help improve click-through rates in search results.', 'prorank-seo'),
                    'how_to_fix' => __('Add a meta description in the post editor or ProRank → On-Page SEO → Titles & Meta.', 'prorank-seo'),
                    'url' => $url,
                ]);
            } else {
                $desc_text = trim($meta_desc->item(0)->nodeValue);
                if (strlen($desc_text) < 50) {
                    $issues[] = $this->issue([
                        'type' => 'short_meta_description',
                        'severity' => 'medium',
                        'category' => 'on_page_seo',
                        'title' => sprintf(
                            /* translators: %s: placeholder value */
                            __('%s meta description is too short', 'prorank-seo'), $page_label),
                        'description' => sprintf(
                            /* translators: %s: numeric value */
                            __('Description is only %d characters. Aim for 120-160.', 'prorank-seo'), strlen($desc_text)),
                        'how_to_fix' => __('Write a more compelling description that summarizes the page content.', 'prorank-seo'),
                        'url' => $url,
                    ]);
                } elseif (strlen($desc_text) > 160) {
                    $issues[] = $this->issue([
                        'type' => 'long_meta_description',
                        'severity' => 'low',
                        'category' => 'on_page_seo',
                        'title' => sprintf(
                            /* translators: %s: placeholder value */
                            __('%s meta description may be truncated', 'prorank-seo'), $page_label),
                        'description' => sprintf(
                            /* translators: %s: numeric value */
                            __('Description is %d characters. Google typically shows ~155-160.', 'prorank-seo'), strlen($desc_text)),
                        'how_to_fix' => __('Consider shortening to under 160 characters.', 'prorank-seo'),
                        'url' => $url,
                    ]);
                }
            }
        }

        if ($this->is_check_enabled($settings, 'headings_structure', true)) {
            // Check H1 tag
            $h1s = $xpath->query('//h1');
            if ($h1s->length === 0) {
                $issues[] = $this->issue([
                    'type' => 'missing_h1',
                    'severity' => 'high',
                    'category' => 'on_page_seo',
                    'title' => sprintf(
                        /* translators: %s: placeholder value */
                        __('%s is missing an H1 heading', 'prorank-seo'), $page_label),
                    'description' => __('The H1 tag is important for SEO and accessibility.', 'prorank-seo'),
                    'how_to_fix' => __('Add an H1 heading that describes the main topic of the page.', 'prorank-seo'),
                    'url' => $url,
                ]);
            } elseif ($h1s->length > 1) {
                $issues[] = $this->issue([
                    'type' => 'multiple_h1',
                    'severity' => 'medium',
                    'category' => 'on_page_seo',
                    'title' => sprintf(/* translators: 1: page name 2: tag count */
                            __('%1$s has %2$d H1 tags', 'prorank-seo'), $page_label, $h1s->length),
                    'description' => __('Having multiple H1 tags can confuse search engines about the main topic.', 'prorank-seo'),
                    'how_to_fix' => __('Use only one H1 tag per page. Convert other H1s to H2 or lower.', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        if ($this->is_check_enabled($settings, 'image_optimization', true)) {
            // Check images for alt text
            $images = $xpath->query('//img');
            $images_without_alt = 0;
            foreach ($images as $img) {
                $alt = $img->getAttribute('alt');
                if (empty(trim($alt))) {
                    $images_without_alt++;
                }
            }
            if ($images_without_alt > 0) {
                $issues[] = $this->issue([
                    'type' => 'images_missing_alt',
                    'severity' => $images_without_alt > 3 ? 'high' : 'medium',
                    'category' => 'on_page_seo',
                    'title' => sprintf(/* translators: 1: image count 2: page name */
                            __('%1$d images missing alt text on %2$s', 'prorank-seo'), $images_without_alt, $page_type),
                    'description' => __('Alt text is important for accessibility and image SEO.', 'prorank-seo'),
                    'how_to_fix' => __('Add descriptive alt text to all images in the Media Library or page editor.', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        // Check Open Graph tags (only for homepage - other pages may not need them)
        if ($page_type === 'homepage' && $this->is_check_enabled($settings, 'open_graph', true)) {
            $og_title = $xpath->query('//meta[@property="og:title"]');
            $og_desc = $xpath->query('//meta[@property="og:description"]');
            $og_image = $xpath->query('//meta[@property="og:image"]');
            $missing_og = [];
            if ($og_title->length === 0) $missing_og[] = 'og:title';
            if ($og_desc->length === 0) $missing_og[] = 'og:description';
            if ($og_image->length === 0) $missing_og[] = 'og:image';

            if (!empty($missing_og)) {
                $issues[] = $this->issue([
                    'type' => 'missing_open_graph',
                    'severity' => 'medium',
                    'category' => 'on_page_seo',
                    'title' => __('Missing Open Graph tags for social sharing', 'prorank-seo'),
                    'description' => sprintf(
                        /* translators: %s: placeholder value */
                        __('Missing: %s. These tags improve how your page looks when shared on social media.', 'prorank-seo'), implode(', ', $missing_og)),
                    'how_to_fix' => __('Configure Open Graph settings in ProRank → On-Page SEO → Social tab.', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        // Check canonical tag
        if ($this->is_check_enabled($settings, 'canonical_tags', true)) {
            $canonical = $xpath->query('//link[@rel="canonical"]/@href');
            if ($canonical->length === 0) {
                $issues[] = $this->issue([
                    'type' => 'missing_canonical',
                    'severity' => 'medium',
                    'category' => 'technical_seo',
                    'title' => sprintf(
                        /* translators: %s: placeholder value */
                        __('%s is missing a canonical tag', 'prorank-seo'), $page_label),
                    'description' => __('Canonical tags help prevent duplicate content issues.', 'prorank-seo'),
                    'how_to_fix' => __('Enable canonical tags in ProRank → Technical SEO settings.', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        // Check viewport meta tag (only check once on homepage)
        if ($page_type === 'homepage' && $this->is_check_enabled($settings, 'mobile_friendly', true)) {
            $viewport = $xpath->query('//meta[@name="viewport"]');
            if ($viewport->length === 0) {
                $issues[] = $this->issue([
                    'type' => 'missing_viewport',
                    'severity' => 'high',
                    'category' => 'technical_seo',
                    'title' => __('Missing viewport meta tag', 'prorank-seo'),
                    'description' => __('The viewport tag is essential for mobile responsiveness.', 'prorank-seo'),
                    'how_to_fix' => __('Your theme should include a viewport meta tag. Check with your theme developer.', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        if ($page_type === 'homepage' && $this->is_check_enabled($settings, 'accessibility', true)) {
            $html_lang = $xpath->query('//html[@lang]');
            if ($html_lang->length === 0) {
                $issues[] = $this->issue([
                    'type' => 'missing_lang',
                    'severity' => 'medium',
                    'category' => 'accessibility',
                    'title' => __('HTML lang attribute is missing', 'prorank-seo'),
                    'description' => __('Declaring a language improves accessibility and assists search engines.', 'prorank-seo'),
                    'how_to_fix' => __('Add a language attribute to the <html> tag (via your theme).', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        // Check for Schema.org markup
        if ($page_type === 'homepage' && $this->is_check_enabled($settings, 'schema_validation', true)) {
            $schema = $xpath->query('//script[@type="application/ld+json"]');
            if ($schema->length === 0) {
                $issues[] = $this->issue([
                    'type' => 'missing_schema',
                    'severity' => 'medium',
                    'category' => 'on_page_seo',
                    'title' => __('No structured data (Schema.org) found', 'prorank-seo'),
                    'description' => __('Schema markup helps search engines understand your content better.', 'prorank-seo'),
                    'how_to_fix' => __('Enable Schema markup in ProRank → On-Page SEO → Schema settings.', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        if ($page_type === 'homepage' && $this->is_check_enabled($settings, 'twitter_cards', true)) {
            $twitter_card = $xpath->query('//meta[@name="twitter:card"]');
            $twitter_title = $xpath->query('//meta[@name="twitter:title"]');
            $twitter_desc = $xpath->query('//meta[@name="twitter:description"]');
            $twitter_image = $xpath->query('//meta[@name="twitter:image"]');
            $missing_twitter = [];
            if ($twitter_card->length === 0) $missing_twitter[] = 'twitter:card';
            if ($twitter_title->length === 0) $missing_twitter[] = 'twitter:title';
            if ($twitter_desc->length === 0) $missing_twitter[] = 'twitter:description';
            if ($twitter_image->length === 0) $missing_twitter[] = 'twitter:image';

            if (!empty($missing_twitter)) {
                $issues[] = $this->issue([
                    'type' => 'missing_twitter_cards',
                    'severity' => 'medium',
                    'category' => 'on_page_seo',
                    'title' => __('Missing Twitter card tags for social sharing', 'prorank-seo'),
                    'description' => sprintf(
                        /* translators: %s: placeholder value */
                        __('Missing: %s. These tags improve previews when shared on X.', 'prorank-seo'), implode(', ', $missing_twitter)),
                    'how_to_fix' => __('Configure Twitter card settings in ProRank → On-Page SEO → Social tab.', 'prorank-seo'),
                    'url' => $url,
                ]);
            }
        }

        return $issues;
    }

    /**
     * Check if a specific audit check type is enabled.
     *
     * @param array  $settings Audit settings.
     * @param string $check_key Check type key.
     * @param bool   $default Default value when missing.
     */
    private function is_check_enabled(array $settings, string $check_key, bool $default = true): bool {
        if (isset($settings['check_types']) && is_array($settings['check_types']) && array_key_exists($check_key, $settings['check_types'])) {
            return (bool) $settings['check_types'][$check_key];
        }

        $fallback_map = [
            'meta_tags' => 'check_meta_tags',
            'schema_validation' => 'check_schema',
            'page_speed' => 'check_performance',
            'core_web_vitals' => 'check_performance',
            'accessibility' => 'check_accessibility',
            'mobile_friendly' => 'check_mobile',
            'https_status' => 'check_security',
            'ssl_certificate' => 'check_security',
            'mixed_content' => 'check_security',
            'security_headers' => 'check_security',
        ];

        if (isset($fallback_map[$check_key]) && array_key_exists($fallback_map[$check_key], $settings)) {
            return (bool) $settings[$fallback_map[$check_key]];
        }

        return $default;
    }

    /**
     * Check XML sitemap
     */
    private function check_sitemap(): array {
        $issues = [];
        $sitemap_url = home_url('/sitemap.xml');

        $response = wp_remote_get($sitemap_url, [
            'timeout' => 5,
            'sslverify' => false,
        ]);

        if (is_wp_error($response)) {
            $issues[] = $this->issue([
                'type' => 'sitemap_error',
                'severity' => 'high',
                'category' => 'technical_seo',
                'title' => __('XML Sitemap is not accessible', 'prorank-seo'),
                'description' => __('Could not fetch your XML sitemap at /sitemap.xml.', 'prorank-seo'),
                'how_to_fix' => __('Ensure XML sitemaps are enabled in ProRank → Technical SEO → Sitemaps.', 'prorank-seo'),
            ]);
            return $issues;
        }

        $status = wp_remote_retrieve_response_code($response);
        if ($status === 404) {
            $issues[] = $this->issue([
                'type' => 'sitemap_missing',
                'severity' => 'high',
                'category' => 'technical_seo',
                'title' => __('XML Sitemap not found (404)', 'prorank-seo'),
                'description' => __('No sitemap was found at /sitemap.xml.', 'prorank-seo'),
                'how_to_fix' => __('Enable XML sitemaps in ProRank → Technical SEO → Sitemaps.', 'prorank-seo'),
            ]);
        } elseif ($status >= 400) {
            $issues[] = $this->issue([
                'type' => 'sitemap_error',
                'severity' => 'high',
                'category' => 'technical_seo',
                'title' => sprintf(
                    /* translators: %s: numeric value */
                    __('XML Sitemap returns HTTP %d', 'prorank-seo'), $status),
                'description' => __('Your sitemap is returning an error.', 'prorank-seo'),
                'how_to_fix' => __('Check your sitemap configuration in ProRank → Technical SEO → Sitemaps.', 'prorank-seo'),
            ]);
        } else {
            $body = wp_remote_retrieve_body($response);
            preg_match_all('/<loc>/i', $body, $matches);
            $url_count = count($matches[0]);

            if ($url_count === 0) {
                $issues[] = $this->issue([
                    'type' => 'sitemap_empty',
                    'severity' => 'medium',
                    'category' => 'technical_seo',
                    'title' => __('XML Sitemap appears to be empty', 'prorank-seo'),
                    'description' => __('Your sitemap exists but contains no URLs.', 'prorank-seo'),
                    'how_to_fix' => __('Check your sitemap settings and ensure content is being included.', 'prorank-seo'),
                ]);
            }
        }

        return $issues;
    }

    /**
     * Check robots.txt
     */
    private function check_robots_txt(): array {
        $issues = [];
        $robots_url = home_url('/robots.txt');

        $response = wp_remote_get($robots_url, [
            'timeout' => 5,
            'sslverify' => false,
        ]);

        if (is_wp_error($response)) {
            return $issues; // Not critical
        }

        $status = wp_remote_retrieve_response_code($response);
        if ($status === 200) {
            $body = wp_remote_retrieve_body($response);

            // Check for common issues
            if (stripos($body, 'Disallow: /') !== false && stripos($body, 'Disallow: / ') === false) {
                // Check if it's blocking everything
                if (preg_match('/Disallow:\s*\/\s*$/m', $body)) {
                    $issues[] = $this->issue([
                        'type' => 'robots_blocking_all',
                        'severity' => 'critical',
                        'category' => 'technical_seo',
                        'title' => __('robots.txt is blocking all crawlers', 'prorank-seo'),
                        'description' => __('Your robots.txt contains "Disallow: /" which blocks search engines from crawling your site.', 'prorank-seo'),
                        'how_to_fix' => __('Edit your robots.txt to allow crawling. Go to ProRank → Technical SEO → Robots & Indexing.', 'prorank-seo'),
                    ]);
                }
            }

            // Check for sitemap reference
            if (stripos($body, 'sitemap:') === false) {
                $issues[] = $this->issue([
                    'type' => 'robots_no_sitemap',
                    'severity' => 'low',
                    'category' => 'technical_seo',
                    'title' => __('robots.txt does not reference your sitemap', 'prorank-seo'),
                    'description' => __('Adding a Sitemap directive helps search engines discover your sitemap.', 'prorank-seo'),
                    /* translators: %s: sitemap URL */
                    'how_to_fix' => sprintf(__('Add "Sitemap: %s" to your robots.txt file.', 'prorank-seo'), home_url('/sitemap.xml')),
                ]);
            }
        }

        return $issues;
    }

    private function issue(array $issue): array {
        $defaults = [
            'id' => (string) wp_generate_uuid4(),
            'type' => '',
            'severity' => 'low',
            'category' => 'content',
            'display_category' => null,
            'title' => '',
            'description' => '',
            'message' => '',
            'url' => home_url('/'),
            'impact' => '',
            'how_to_fix' => '',
            'reference' => null,
        ];

        $merged = array_merge($defaults, $issue);

        if (empty($merged['display_category'])) {
            $merged['display_category'] = $merged['category'];
        }

        if (empty($merged['message'])) {
            $merged['message'] = $merged['description'] ?: $merged['title'];
        }

        $allowed = ['critical', 'high', 'medium', 'low', 'passed'];
        if (!in_array($merged['severity'], $allowed, true)) {
            $merged['severity'] = 'low';
        }

        $merged['category'] = (string) $merged['category'];
        $merged['display_category'] = (string) $merged['display_category'];
        $merged['title'] = (string) $merged['title'];
        $merged['description'] = (string) $merged['description'];
        $merged['message'] = (string) $merged['message'];
        $merged['how_to_fix'] = (string) $merged['how_to_fix'];

        return $merged;
    }

    private function count_severities(array $issues): array {
        $counts = [
            'critical' => 0,
            'high' => 0,
            'medium' => 0,
            'low' => 0,
            'warning' => 0,
            'total' => 0,
        ];

        foreach ($issues as $issue) {
            $severity = (string) ($issue['severity'] ?? 'low');
            if (!isset($counts[$severity])) {
                $severity = 'low';
            }
            $counts[$severity]++;
            $counts['total']++;
        }

        // UI often treats low as "warning"
        $counts['warning'] = $counts['low'];

        return $counts;
    }

    private function calculate_score(array $counts): int {
        $score = 100;
        $score -= min(60, $counts['critical'] * 15);
        $score -= min(40, $counts['high'] * 8);
        $score -= min(25, $counts['medium'] * 4);
        $score -= min(15, $counts['low'] * 2);

        return max(0, min(100, $score));
    }

    private function get_issue_counts(string $audit_id): array {
        global $wpdb;

        $counts = [
            'critical' => 0,
            'high' => 0,
            'medium' => 0,
            'low' => 0,
            'warning' => 0,
            'total' => 0,
        ];

        if (!$this->table_exists($this->audit_issues_table)) {
            return $counts;
        }

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $rows = $wpdb->get_results(
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->prepare(
                "SELECT severity, COUNT(*) as cnt
                 FROM {$this->audit_issues_table}
                 WHERE audit_id = %s
                 GROUP BY severity",
                $audit_id
            )
        );

        foreach ($rows as $row) {
            $sev = (string) $row->severity;
            if (isset($counts[$sev])) {
                $counts[$sev] = (int) $row->cnt;
            }
        }

        $counts['total'] = $counts['critical'] + $counts['high'] + $counts['medium'] + $counts['low'];
        $counts['warning'] = $counts['low'];

        return $counts;
    }

    private function has_column(string $table, string $column): bool {
        $cache_key = $table . ':' . $column;
        if (array_key_exists($cache_key, $this->column_cache)) {
            return $this->column_cache[$cache_key];
        }

        global $wpdb;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table/column names are controlled internally
        $exists = $wpdb->get_var($wpdb->prepare("SHOW COLUMNS FROM {$table} LIKE %s", $column));
        $this->column_cache[$cache_key] = !empty($exists);

        return $this->column_cache[$cache_key];
    }

    private function maybe_create_tables(): void {
        global $wpdb;

        $tables_exist = $this->table_exists($this->audits_table)
            && $this->table_exists($this->audit_urls_table)
            && $this->table_exists($this->audit_issues_table);

        if ($tables_exist) {
            $needs_upgrade = false;
            if (!$this->has_column($this->audits_table, 'progress') || !$this->has_column($this->audits_table, 'pages_crawled')) {
                $needs_upgrade = true;
            }
            if (!$needs_upgrade) {
                return;
            }
        }

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $charset_collate = $wpdb->get_charset_collate();

        $sql1 = "CREATE TABLE {$this->audits_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(50) NOT NULL,
            status enum('idle','crawling','checking','completed','failed','stopped','stopping') DEFAULT 'idle',
            started_at datetime DEFAULT NULL,
            completed_at datetime DEFAULT NULL,
            total_urls int(11) DEFAULT 0,
            score int(3) DEFAULT NULL,
            stats longtext,
            progress longtext,
            options longtext,
            pages_crawled int(11) DEFAULT 0,
            error text,
            PRIMARY KEY  (id),
            UNIQUE KEY audit_id (audit_id),
            KEY status (status),
            KEY started_at (started_at)
        ) $charset_collate;";

        $sql2 = "CREATE TABLE {$this->audit_urls_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(50) NOT NULL,
            url varchar(500) NOT NULL,
            status enum('pending','checking','checked') DEFAULT 'pending',
            issues_count int(11) DEFAULT 0,
            warnings_count int(11) DEFAULT 0,
            passed_count int(11) DEFAULT 0,
            score int(3) DEFAULT NULL,
            created_at datetime DEFAULT NULL,
            checked_at datetime DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY audit_id (audit_id),
            KEY status (status),
            KEY url (url(255))
        ) $charset_collate;";

        $sql3 = "CREATE TABLE {$this->audit_issues_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            audit_id varchar(50) NOT NULL,
            url_id bigint(20) unsigned NOT NULL,
            type varchar(50) NOT NULL,
            severity enum('critical','high','medium','low','passed') DEFAULT 'medium',
            message text,
            data longtext,
            created_at datetime DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY audit_id (audit_id),
            KEY url_id (url_id),
            KEY type (type),
            KEY severity (severity)
        ) $charset_collate;";

        dbDelta($sql1);
        dbDelta($sql2);
        dbDelta($sql3);

        // Clear column cache after schema updates.
        $this->column_cache = [];
    }

    private function table_exists(string $table): bool {
        global $wpdb;
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $found = $wpdb->get_var($wpdb->prepare('SHOW TABLES LIKE %s', $table));
        return $found === $table;
    }
}
