<?php
/**
 * Meta Box Handler
 *
 * Handles the SEO meta box in the post editor.
 *
 * @package ProRank\SEO\Admin
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;

/**
 * MetaBox class
 */
class MetaBox {
    /**
     * Plugin instance
     *
     * @var Plugin
     */
    private Plugin $plugin;

    /**
     * Constructor
     *
     * @param Plugin $plugin Plugin instance.
     */
    public function __construct(Plugin $plugin) {
        $this->plugin = $plugin;
    }

    /**
     * Initialize meta box
     *
     * @return void
     */
    public function init(): void {
        add_action('add_meta_boxes', [$this, 'add_meta_box']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }

    /**
     * Add meta box to post editor
     *
     * @return void
     */
    public function add_meta_box(): void {
        $screen = get_current_screen();
        
        // Only add to post edit screens
        if (!$screen || $screen->base !== 'post') {
            return;
        }

        // Get publicly queryable post types
        $post_types = get_post_types(['public' => true], 'names');
        $titles_meta_settings = get_option('prorank_seo_titles_meta', []);

        $post_types = array_values(array_filter($post_types, function ($post_type) use ($titles_meta_settings) {
            $type_settings = is_array($titles_meta_settings) ? ($titles_meta_settings["post_type_{$post_type}"] ?? []) : [];
            return !array_key_exists('show_meta_box', $type_settings) || $type_settings['show_meta_box'] !== false;
        }));

        if (empty($post_types)) {
            return;
        }
        
        add_meta_box(
            'prorank-seo-meta-box',
            __('ProRank SEO', 'prorank-seo'),
            [$this, 'render_meta_box'],
            $post_types,
            'normal',
            'high'
        );
    }

    /**
     * Render meta box content
     *
     * @param \WP_Post $post Current post object.
     * @return void
     */
    public function render_meta_box(\WP_Post $post): void {
        // Output container for React app
        echo '<div id="prorank-seo-metabox-root"></div>';
        
        // Add nonce field for security
        wp_nonce_field('prorank_seo_metabox', 'prorank_seo_metabox_nonce');
    }

    /**
     * Enqueue meta box assets
     *
     * @param string $hook Current admin page hook.
     * @return void
     */
    public function enqueue_assets(string $hook): void {
        // Only load on post edit screens
        if (!in_array($hook, ['post.php', 'post-new.php'], true)) {
            return;
        }

        $screen = get_current_screen();
        if (!$screen || $screen->base !== 'post') {
            return;
        }

        // Check if we're on a supported post type
        $post_type = $screen->post_type;
        $public_post_types = get_post_types(['public' => true], 'names');
        
        if (!in_array($post_type, $public_post_types, true)) {
            return;
        }

        $titles_meta_settings = get_option('prorank_seo_titles_meta', []);
        $type_settings = is_array($titles_meta_settings) ? ($titles_meta_settings["post_type_{$post_type}"] ?? []) : [];
        if (array_key_exists('show_meta_box', $type_settings) && $type_settings['show_meta_box'] === false) {
            return;
        }

        // Enqueue meta box script from build directory
        $script_url = PRORANK_PLUGIN_URL . 'build/meta-box.js';

        // Get version from asset file if available
        $asset_file = PRORANK_PLUGIN_DIR . 'build/meta-box.asset.php';
        $asset_data = file_exists($asset_file) ? require $asset_file : [];
        $dependencies = $asset_data['dependencies'] ?? ['wp-element', 'wp-components', 'wp-i18n', 'wp-data', 'wp-api-fetch', 'wp-dom-ready', 'lodash'];
        $version = $asset_data['version'] ?? PRORANK_VERSION;

        wp_enqueue_script(
            'prorank-seo-meta-box',
            $script_url,
            $dependencies,
            $version,
            true
        );

        // Enqueue meta box CSS
        $css_url = PRORANK_PLUGIN_URL . 'build/meta-box.css';
        wp_enqueue_style(
            'prorank-seo-meta-box',
            $css_url,
            [],
            $version
        );

        // Get current post
        global $post;
        $post_id = $post ? $post->ID : 0;

        // Localize script data
        wp_localize_script(
            'prorank-seo-meta-box',
            'prorankMetaboxData',
            [
                'postId' => $post_id,
                'nonce' => wp_create_nonce('prorank_seo_metabox'),
                'apiUrl' => rest_url('prorank-seo/v1'),
            ]
        );

        // Compatibility + safety: some bundles rely on window.prorankSettings being present.
        // Without it, checks like `window.prorankSettings?.license?.tier !== 'free'` become
        // true when the object is missing, which wrongly enables Pro-only UI and triggers 404s.
        $inline = 'window.prorankSettings = window.prorankSettings || {};';
        $inline .= 'window.prorankSettings.adminUrl = ' . wp_json_encode(admin_url()) . ';';
        wp_add_inline_script('prorank-seo-meta-box', $inline, 'before');

        // Set script translations
        wp_set_script_translations(
            'prorank-seo-meta-box',
            'prorank-seo',
            PRORANK_PLUGIN_DIR . 'languages'
        );
    }
}
