<?php
/**
 * Admin Menu Handler
 *
 * Utilizes the MenuRegistry service to construct the WordPress admin menu
 * following the 11-item IA structure from ProRank_Combined_Reference_v1.md §3.1.
 *
 * @package ProRank\Admin
 * @since   0.1.0
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\SEO\Plugin;
use ProRank\SEO\Core\MenuRegistry;
use ProRank\SEO\Core\ModuleManager;
use ProRank\SEO\Admin\Pages\SupportAccountPage;

/**
 * MenuHandler class
 *
 * Registers admin menu items with conditional visibility
 */
class MenuHandler {

    /**
     * Plugin instance
     *
     * @var Plugin
     */
    private Plugin $plugin;

    /**
     * Support & Account page instance
     *
     * @var SupportAccountPage|null
     */
    private ?SupportAccountPage $support_account_page = null;



    /**
     * Constructor
     *
     * @param Plugin $plugin Plugin instance
     */
    public function __construct(Plugin $plugin) {
        $this->plugin = $plugin;

        // Initialize pages
        $this->support_account_page = new SupportAccountPage($this->plugin->license());
        add_action('admin_enqueue_scripts', [$this->support_account_page, 'enqueue_assets']);
    }

    /**
     * Initialize menu registration
     *
     * @return void
     */
    public function init(): void {
        // Handle OAuth callbacks early
        add_action('admin_init', [$this, 'handle_oauth_callbacks']);

        $menu_registry = $this->plugin->menus();

        // Register top-level ProRank SEO menu
        $menu_registry->register_top_level(
            'prorank-seo',
            'ProRank SEO Dashboard',
            'ProRank SEO',
            'manage_options',
            [$this, 'render_dashboard_page'],
            $this->get_menu_icon(),
            null, // No condition for main menu
            26
        );

        // 1. Dashboard (always visible as main page)
        // Register with same slug as parent to replace auto-generated submenu
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-seo', // Same as parent for first item
            'Dashboard',
            'Dashboard',
            'manage_options',
            [$this, 'render_dashboard_page'],
            null,
            0 // Position 0 to ensure it's first
        );

        // 2. On-Page SEO (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-on-page-seo',
            'On-Page SEO',
            'On-Page SEO',
            'manage_options',
            [$this, 'render_on_page_seo_page'],
            null, // No special conditions for free tier
            10
        );

        // Hidden: Bulk Edit page (only accessible via posts list bulk action)
        // Register directly with WordPress using null parent to avoid menu entry
        add_action('admin_menu', function() {
            add_submenu_page(
                null, // null parent = no menu entry
                __('Bulk Edit SEO', 'prorank-seo'),
                __('Bulk Edit SEO', 'prorank-seo'),
                'edit_posts',
                'prorank-bulk-edit',
                [$this, 'render_bulk_edit_page']
            );
        }, 20);

        // 3. Performance (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-performance',
            'Performance',
            'Performance',
            'manage_options',
            [$this, 'render_performance_page'],
            null, // Base features are free
            20
        );

        // 4. Technical SEO (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-technical-seo',
            'Technical SEO',
            'Technical SEO',
            'manage_options',
            [$this, 'render_technical_seo_page'],
            null, // Base features are free
            30
        );

        // 9. Site Audit & Tools (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-site-audit',
            'Site Audit & Tools',
            'Site Audit & Tools',
            'manage_options',
            [$this, 'render_site_audit_page'],
            null, // Base features are free
            80
        );

        // 11. Support & Account (FREE)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-support-account',
            'Support & Account',
            'Support & Account',
            'manage_options',
            [$this->support_account_page, 'render'],
            null, // Always visible
            100
        );

        // 12. Upgrade (FREE CTA)
        $menu_registry->register_sub_menu(
            'prorank-seo',
            'prorank-upgrade',
            'Upgrade',
            'Upgrade',
            'manage_options',
            [$this, 'render_upgrade_page'],
            null,
            110
        );

        // Actually register all menus with WordPress
        $menu_registry->register_menus();
    }

    /**
     * Get menu icon
     *
     * Returns the ProRank favicon as a data URI for WordPress admin menu.
     *
     * @return string
     */
    private function get_menu_icon(): string {
        $icon_path = PRORANK_PLUGIN_DIR . 'assets/prorank.svg';
        if ( file_exists( $icon_path ) ) {
            $svg = prorank_read_file( $icon_path );
            if ( $svg !== false ) {
                return 'data:image/svg+xml;base64,' . base64_encode( $svg );
            }
        }

        // Fallback: simple hexagonal icon.
        $fallback = '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="none"><path d="M12 2L4 6.5V17.5L12 22L20 17.5V6.5L12 2Z" fill="#a7aaad" fill-opacity="0.15" stroke="#a7aaad" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>';
        return 'data:image/svg+xml;base64,' . base64_encode( $fallback );
    }

    /**
     * Render dashboard page
     *
     * @return void
     */
    public function render_dashboard_page(): void {
        echo '<div id="prorank-seo-dashboard" class="wrap prorank-seo-admin-page">';
        echo '<h1>' . esc_html('ProRank SEO Dashboard') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render upgrade page
     *
     * @return void
     */
    public function render_upgrade_page(): void {
        $upgrade_url = defined('PRORANK_PRICING_URL') ? PRORANK_PRICING_URL : 'https://prorank.io/pricing';
        $css = '
            .prorank-upgrade-page {
                margin: 20px 20px 20px 0;
            }
            .prorank-upgrade-hero {
                background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
                border-radius: 16px;
                padding: 80px 40px;
                text-align: center;
                box-shadow: 0 18px 45px rgba(29, 78, 216, 0.35);
                max-width: 900px;
                margin: 0 auto;
            }
            .prorank-upgrade-hero h1 {
                color: #ffffff;
                font-size: 42px;
                font-weight: 700;
                margin: 0 0 16px 0;
                padding: 0;
                line-height: 1.2;
            }
            .prorank-upgrade-hero p {
                color: rgba(255, 255, 255, 0.85);
                font-size: 16px;
                line-height: 1.7;
                max-width: 640px;
                margin: 0 auto 28px auto;
            }
            .prorank-upgrade-btn {
                display: inline-flex;
                align-items: center;
                justify-content: center;
                background: #ffffff;
                color: #1d4ed8;
                font-size: 16px;
                font-weight: 600;
                padding: 14px 32px;
                border-radius: 999px;
                text-decoration: none;
                transition: transform 0.2s ease, box-shadow 0.2s ease;
                box-shadow: 0 8px 20px rgba(15, 23, 42, 0.18);
            }
            .prorank-upgrade-btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 12px 28px rgba(15, 23, 42, 0.22);
            }
            @media (max-width: 782px) {
                .prorank-upgrade-hero {
                    padding: 56px 24px;
                }
                .prorank-upgrade-hero h1 {
                    font-size: 32px;
                }
                .prorank-upgrade-hero p {
                    font-size: 15px;
                }
            }
        ';

        wp_register_style('prorank-upgrade', false, [], PRORANK_VERSION);
        wp_enqueue_style('prorank-upgrade');
        wp_add_inline_style('prorank-upgrade', prorank_sanitize_inline_css($css));

        echo '<div class="wrap prorank-seo-admin-page prorank-upgrade-page">';
        echo '<div class="prorank-upgrade-hero">';
        echo '<h1>' . esc_html__('Upgrade to ProRank Premium', 'prorank-seo') . '</h1>';
        echo '<p>' . esc_html__('Choose the plan that fits your needs and unlock cloud image optimization, advanced performance tools, advanced SEO tools, Core Web Vitals monitoring, and automation.', 'prorank-seo') . '</p>';
        echo '<a class="prorank-upgrade-btn" href="' . esc_url($upgrade_url) . '" target="_blank" rel="noopener noreferrer">' . esc_html__('View Plans', 'prorank-seo') . '</a>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render On-Page SEO page
     *
     * @return void
     */
    public function render_on_page_seo_page(): void {
        echo '<div id="prorank-seo-on-page" class="wrap prorank-seo-admin-page prorank-on-page-seo-page">';
        echo '<h1>' . esc_html('On-Page SEO') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Performance page
     *
     * @return void
     */
    public function render_performance_page(): void {
        echo '<div id="prorank-seo-performance" class="wrap prorank-seo-admin-page">';
        echo '<h1>' . esc_html('Performance') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Technical SEO page
     *
     * @return void
     */
    public function render_technical_seo_page(): void {
        echo '<div id="prorank-seo-technical" class="wrap prorank-seo-admin-page prorank-technical-seo-page">';
        echo '<h1>' . esc_html('Technical SEO') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';

        // Debug: Check if scripts are enqueued
        global $wp_scripts;
        if (isset($wp_scripts->registered['prorank-seo-admin'])) {
            echo '<!-- ProRank SEO Admin script is registered -->';
        } else {
            echo '<!-- WARNING: ProRank SEO Admin script is NOT registered -->';
        }
        echo '</div>';
    }


    /**
     * Handle OAuth callbacks early in admin_init
     *
     * Note: OAuth callbacks are only used by premium Analytics feature.
     * This method is kept as a stub for compatibility.
     *
     * @return void
     */
    public function handle_oauth_callbacks(): void {
        // OAuth callbacks are premium-only (Analytics feature)
        // No action needed in free version
    }

    /**
     * Render Site Audit & Tools page
     *
     * @return void
     */
    public function render_site_audit_page(): void {
        echo '<div id="prorank-seo-site-audit" class="wrap prorank-seo-admin-page prorank-site-audit-page">';

        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Support & Account page
     *
     * @return void
     */
    public function render_support_page(): void {
        echo '<div id="prorank-seo-support" class="wrap prorank-seo-admin-page">';
        echo '<h1>' . esc_html('Support & Account') . '</h1>';
        // Add the container that React app is looking for
        echo '<div class="prorank-admin-container" id="prorank-admin-root">';
        echo '<!-- React component will mount here -->';
        echo '<noscript>' . esc_html('JavaScript is required to view this page.') . '</noscript>';
        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Import/Export page
     *
     * @return void
     */
    public function render_import_export_page(): void {
        // Always render the React container - React handles locked state per feature
        echo '<div id="prorank-seo-import-export" class="wrap prorank-seo-admin-page prorank-import-export-page">';
        echo '<div class="prorank-admin-container" id="prorank-admin-root"></div>';
        echo '</div>';
    }

    /**
     * Render Bulk Edit SEO page
     *
     * @return void
     */
    public function render_bulk_edit_page(): void {
        // Get post IDs from URL
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only URL args.
        $post_ids_raw = isset($_GET['post_ids']) ? sanitize_text_field( wp_unslash( $_GET['post_ids'] ) ) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only URL args.
        $post_type = isset($_GET['post_type']) ? sanitize_key( wp_unslash( $_GET['post_type'] ) ) : 'post';
        $post_ids = array_filter(array_map('intval', explode(',', $post_ids_raw)));

        echo '<div class="wrap prorank-seo-admin-page prorank-bulk-edit-page">';
        echo '<h1>' . esc_html__('Bulk Edit SEO', 'prorank-seo') . '</h1>';

        if (empty($post_ids)) {
            echo '<div class="notice notice-warning"><p>' . esc_html__('No posts selected. Please select posts from the posts list and use the "Edit SEO Settings" bulk action.', 'prorank-seo') . '</p></div>';
            echo '<p><a href="' . esc_url(admin_url('edit.php?post_type=' . $post_type)) . '" class="button">' . esc_html__('Go to Posts', 'prorank-seo') . '</a></p>';
            echo '</div>';
            return;
        }

        // Handle form submission
        if (isset($_POST['prorank_bulk_edit_submit']) && check_admin_referer('prorank_bulk_edit_nonce', 'prorank_bulk_edit_nonce')) {
            $updated = 0;
            foreach ($post_ids as $post_id) {
                if (!current_user_can('edit_post', $post_id)) {
                    continue;
                }
                $title_key = 'prorank_seo_title_' . $post_id;
                $desc_key = 'prorank_seo_desc_' . $post_id;
                $keyword_key = 'prorank_focus_keyword_' . $post_id;

                if (isset($_POST[$title_key])) {
                    $title_value = trim((string) sanitize_text_field(wp_unslash($_POST[$title_key])));
                    if ($title_value !== '') {
                        update_post_meta($post_id, '_prorank_seo_title', $title_value);
                    }
                }
                if (isset($_POST[$desc_key])) {
                    $desc_value = trim((string) sanitize_textarea_field(wp_unslash($_POST[$desc_key])));
                    if ($desc_value !== '') {
                        update_post_meta($post_id, '_prorank_seo_description', $desc_value);
                    }
                }
                if (isset($_POST[$keyword_key])) {
                    $keyword_value = trim((string) sanitize_text_field(wp_unslash($_POST[$keyword_key])));
                    if ($keyword_value !== '') {
                        update_post_meta($post_id, '_prorank_seo_focus_keyword', $keyword_value);
                    }
                }
                $updated++;
            }
            /* translators: %d: number of posts updated */
            echo '<div class="notice notice-success is-dismissible"><p>' . sprintf(esc_html__('Updated SEO settings for %d posts.', 'prorank-seo'), (int) $updated) . '</p></div>';
        }

        /* translators: %d: number of posts being edited */
        echo '<p>' . sprintf(esc_html__('Editing SEO settings for %d posts.', 'prorank-seo'), count($post_ids)) . ' ';
        echo '<a href="' . esc_url(admin_url('edit.php?post_type=' . $post_type)) . '">' . esc_html__('Back to posts list', 'prorank-seo') . '</a></p>';

        echo '<form method="post">';
        wp_nonce_field('prorank_bulk_edit_nonce', 'prorank_bulk_edit_nonce');

        echo '<table class="wp-list-table widefat fixed striped">';
        echo '<thead><tr>';
        echo '<th style="width:25%">' . esc_html__('Post Title', 'prorank-seo') . '</th>';
        echo '<th style="width:15%">' . esc_html__('Focus Keyword', 'prorank-seo') . '</th>';
        echo '<th style="width:25%">' . esc_html__('SEO Title', 'prorank-seo') . ' <small>(50-60)</small></th>';
        echo '<th style="width:35%">' . esc_html__('Meta Description', 'prorank-seo') . ' <small>(120-155)</small></th>';
        echo '</tr></thead>';
        echo '<tbody>';

        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if (!$post) continue;

            $seo_title = $this->get_meta_with_fallback($post_id, '_prorank_seo_title', [
                '_prorank_meta_title',
                '_prorank_title',
            ]);
            $seo_desc = $this->get_meta_with_fallback($post_id, '_prorank_seo_description', [
                '_prorank_meta_description',
                '_prorank_description',
            ]);
            $focus_keyword = get_post_meta($post_id, '_prorank_seo_focus_keyword', true);

            $title_placeholder = '';
            $desc_placeholder = '';
            if ($seo_title === '') {
                $title_placeholder = $this->safe_substr(wp_strip_all_tags($post->post_title), 0, 70);
            }
            if ($seo_desc === '') {
                $desc_source = $post->post_excerpt !== '' ? $post->post_excerpt : $post->post_content;
                $desc_placeholder = $this->safe_substr(wp_strip_all_tags($desc_source), 0, 160);
            }

            $title_data = $this->get_bulk_progress_data($seo_title, 70, 50, 60, 30, 70);
            $desc_data = $this->get_bulk_progress_data($seo_desc, 170, 120, 155, 80, 170);
            echo '<tr>';
            echo '<td><strong><a href="' . esc_url(get_edit_post_link($post_id)) . '">' . esc_html($post->post_title) . '</a></strong></td>';
            echo '<td>';
            echo '<div class="prorank-bulk-edit-field">';
            echo '<input type="text" class="prorank-bulk-input prorank-focus-keyword" name="prorank_focus_keyword_' . esc_attr($post_id) . '" value="' . esc_attr($focus_keyword) . '" style="width:100%">';
            echo '</div>';
            echo '</td>';
            echo '<td>';
            echo '<div class="prorank-bulk-edit-field" data-max="' . esc_attr($title_data['max']) . '" data-optimal-min="50" data-optimal-max="60" data-acceptable-min="30" data-acceptable-max="70">';
            echo '<input type="text" class="prorank-bulk-input prorank-seo-title" name="prorank_seo_title_' . esc_attr($post_id) . '" value="' . esc_attr($seo_title) . '" maxlength="70" style="width:100%"' . ($title_placeholder !== '' ? ' placeholder="' . esc_attr($title_placeholder) . '"' : '') . '>';
            echo '<div class="prorank-field-meta">';
            echo '<span class="prorank-char-count">' . esc_html($title_data['length']) . '/' . esc_html($title_data['max']) . '</span>';
            echo '</div>';
            echo '<div class="prorank-progress-bar ' . esc_attr($title_data['class']) . '" role="progressbar" aria-valuenow="' . esc_attr($title_data['length']) . '" aria-valuemin="0" aria-valuemax="' . esc_attr($title_data['max']) . '">';
            echo '<div class="progress-fill" style="width: ' . esc_attr($title_data['percent']) . '%"></div>';
            echo '</div>';
            echo '</div>';
            echo '</td>';
            echo '<td>';
            echo '<div class="prorank-bulk-edit-field" data-max="' . esc_attr($desc_data['max']) . '" data-optimal-min="120" data-optimal-max="155" data-acceptable-min="80" data-acceptable-max="170">';
            echo '<textarea class="prorank-bulk-input prorank-seo-desc" name="prorank_seo_desc_' . esc_attr($post_id) . '" rows="2" maxlength="200" style="width:100%"' . ($desc_placeholder !== '' ? ' placeholder="' . esc_attr($desc_placeholder) . '"' : '') . '>' . esc_textarea($seo_desc) . '</textarea>';
            echo '<div class="prorank-field-meta">';
            echo '<span class="prorank-char-count">' . esc_html($desc_data['length']) . '/' . esc_html($desc_data['max']) . '</span>';
            echo '</div>';
            echo '<div class="prorank-progress-bar ' . esc_attr($desc_data['class']) . '" role="progressbar" aria-valuenow="' . esc_attr($desc_data['length']) . '" aria-valuemin="0" aria-valuemax="' . esc_attr($desc_data['max']) . '">';
            echo '<div class="progress-fill" style="width: ' . esc_attr($desc_data['percent']) . '%"></div>';
            echo '</div>';
            echo '</div>';
            echo '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
        echo '<p class="submit"><input type="submit" name="prorank_bulk_edit_submit" class="button button-primary" value="' . esc_attr__('Save All Changes', 'prorank-seo') . '"></p>';
        echo '</form>';
        echo '</div>';
    }

    /**
     * Build bulk edit progress data for length feedback.
     *
     * @param string $value           Current field value.
     * @param int    $max             Max length for progress calculations.
     * @param int    $optimal_min     Optimal minimum length.
     * @param int    $optimal_max     Optimal maximum length.
     * @param int    $acceptable_min  Acceptable minimum length.
     * @param int    $acceptable_max  Acceptable maximum length.
     * @return array{length:int, percent:float, class:string, max:int}
     */
    private function get_bulk_progress_data(string $value, int $max, int $optimal_min, int $optimal_max, int $acceptable_min, int $acceptable_max): array {
        $clean_value = wp_strip_all_tags($value);
        $length = $this->safe_strlen($clean_value);
        $percent = $max > 0 ? min(100, ($length / $max) * 100) : 0;

        if ($length === 0) {
            $class = 'empty';
        } elseif ($length < $acceptable_min || $length > $acceptable_max) {
            $class = 'over';
        } elseif ($length >= $optimal_min && $length <= $optimal_max) {
            $class = 'good';
        } else {
            $class = 'warning';
        }

        return [
            'length' => $length,
            'percent' => $percent,
            'class' => $class,
            'max' => $max,
        ];
    }

    /**
     * Read a meta value with fallback keys for legacy data.
     *
     * @param int    $post_id   Post ID.
     * @param string $primary   Primary meta key.
     * @param array  $fallbacks Fallback meta keys.
     * @return string
     */
    private function get_meta_with_fallback(int $post_id, string $primary, array $fallbacks): string {
        $value = get_post_meta($post_id, $primary, true);
        if ($value !== '') {
            return (string) $value;
        }

        foreach ($fallbacks as $fallback) {
            $value = get_post_meta($post_id, $fallback, true);
            if ($value !== '') {
                return (string) $value;
            }
        }

        return '';
    }

    /**
     * Safe string length helper (multibyte aware).
     *
     * @param string $value Input string.
     * @return int
     */
    private function safe_strlen(string $value): int {
        if (function_exists('mb_strlen')) {
            return (int) mb_strlen($value);
        }

        return (int) strlen($value);
    }

    /**
     * Safe substring helper (multibyte aware).
     *
     * @param string $value Input string.
     * @param int    $start Start offset.
     * @param int    $length Length to return.
     * @return string
     */
    private function safe_substr(string $value, int $start, int $length): string {
        if (function_exists('mb_substr')) {
            return (string) mb_substr($value, $start, $length);
        }

        return (string) substr($value, $start, $length);
    }

}
