<?php

declare(strict_types=1);

/**
 * Internal Linking Style Overrides
 *
 * Injects clean styles via wp_add_inline_style to override boxy appearance
 */

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

class InternalLinkingStyles {

    public static function init() {
        add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_inline_assets'], 99);
    }

    public static function enqueue_inline_assets() {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only page check.
        $page = isset($_GET['page']) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
        if ($page === '' || strpos($page, 'prorank-seo') === false) {
            return;
        }

        // Register placeholder handles for inline assets
        wp_register_style('prorank-internal-linking-clean', false, [], PRORANK_VERSION);
        wp_enqueue_style('prorank-internal-linking-clean');

        wp_register_script('prorank-internal-linking-runtime', '', [], PRORANK_VERSION, true);
        wp_enqueue_script('prorank-internal-linking-runtime');

        // Add inline CSS
        wp_add_inline_style('prorank-internal-linking-clean', prorank_sanitize_inline_css(self::get_inline_css()));

        // Add inline JS
        wp_add_inline_script('prorank-internal-linking-runtime', prorank_sanitize_inline_js(self::get_inline_js()));
    }

    private static function get_inline_css(): string {
        return '
        /* Remove ALL box effects from Internal Linking stats */
        .prorank-internal-linking-dashboard .prorank-dashboard-stats {
            display: grid !important;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr)) !important;
            gap: 20px !important;
            margin-bottom: 30px !important;
        }

        /* Clean stat cards - NO BOXES */
        .prorank-internal-linking-dashboard .prorank-stat-card,
        .prorank-internal-linking-dashboard .components-card.prorank-stat-card {
            background: transparent !important;
            border: none !important;
            box-shadow: none !important;
            padding: 0 !important;
        }

        .prorank-internal-linking-dashboard .prorank-stat-card .components-card__body {
            padding: 16px 20px !important;
            background: #f8f9fa !important;
            border-radius: 8px !important;
            border: 1px solid #e0e0e0 !important;
            box-shadow: none !important;
            display: flex !important;
            align-items: center !important;
            gap: 15px !important;
            min-height: 90px !important;
        }

        .prorank-internal-linking-dashboard .prorank-stat-card:hover .components-card__body {
            background: #ffffff !important;
            border-color: #0073aa !important;
        }

        /* Stat icons - smaller and cleaner */
        .prorank-stat-icon {
            width: 40px !important;
            height: 40px !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            border-radius: 8px !important;
            background: #e3f2fd !important;
            flex-shrink: 0 !important;
        }

        .prorank-stat-icon svg {
            width: 20px !important;
            height: 20px !important;
            color: #0073aa !important;
        }

        .prorank-stat-icon--warning {
            background: #fff3cd !important;
        }

        .prorank-stat-icon--warning svg {
            color: #856404 !important;
        }

        .prorank-stat-icon--error {
            background: #f8d7da !important;
        }

        .prorank-stat-icon--error svg {
            color: #721c24 !important;
        }

        .prorank-stat-icon--success {
            background: #d4edda !important;
        }

        .prorank-stat-icon--success svg {
            color: #155724 !important;
        }

        /* Stat content */
        .prorank-stat-content {
            flex: 1 !important;
        }

        .prorank-stat-label {
            font-size: 11px !important;
            font-weight: 600 !important;
            color: #666 !important;
            text-transform: uppercase !important;
            letter-spacing: 0.5px !important;
            margin: 0 0 5px 0 !important;
        }

        .prorank-stat-value {
            font-size: 24px !important;
            font-weight: 700 !important;
            color: #23282d !important;
            line-height: 1 !important;
            margin: 0 !important;
        }

        /* Clean up other cards */
        .prorank-link-health-card,
        .prorank-advanced-features-card,
        .prorank-domains-card,
        .prorank-issues-card {
            box-shadow: 0 1px 3px rgba(0,0,0,0.05) !important;
            border: 1px solid #e0e0e0 !important;
            border-radius: 8px !important;
        }

        .prorank-link-health-card .components-card__header,
        .prorank-advanced-features-card .components-card__header,
        .prorank-domains-card .components-card__header,
        .prorank-issues-card .components-card__header {
            background: #f5f5f5 !important;
            border-bottom: 1px solid #e0e0e0 !important;
            padding: 12px 16px !important;
        }

        .prorank-link-health-card h3,
        .prorank-advanced-features-card h3,
        .prorank-domains-card h3,
        .prorank-issues-card h3 {
            margin: 0 !important;
            font-size: 14px !important;
            font-weight: 600 !important;
            color: #23282d !important;
        }

        /* Progress bars */
        .prorank-health-metric {
            padding: 12px 0 !important;
            border-bottom: 1px solid #f0f0f0 !important;
        }

        .prorank-health-metric:last-child {
            border-bottom: none !important;
        }

        /* Remove extra shadows and borders */
        .prorank-internal-linking-dashboard .components-card {
            box-shadow: 0 1px 3px rgba(0,0,0,0.05) !important;
        }

        .prorank-internal-linking-dashboard .components-notice {
            box-shadow: none !important;
            border-left-width: 3px !important;
        }

        /* Feature badges */
        .prorank-feature-badge {
            display: inline-flex !important;
            align-items: center !important;
            gap: 4px !important;
            padding: 4px 10px !important;
            background: white !important;
            border-radius: 15px !important;
            font-size: 11px !important;
            font-weight: 500 !important;
            border: 1px solid #ddd !important;
        }

        .prorank-feature-badge svg {
            width: 14px !important;
            height: 14px !important;
        }
        ';
    }

    private static function get_inline_js(): string {
        return '
        // Runtime fix to ensure tabs show all features
        (function() {
            // Check if we are on the internal linking tab
            var checkInterval = setInterval(function() {
                var tabPanels = document.querySelectorAll(".components-tab-panel__tab-content");
                var internalLinkingContent = Array.from(tabPanels).find(function(panel) {
                    return panel.innerHTML.includes("Internal Linking") ||
                           panel.innerHTML.includes("prorank-internal-linking");
                });

                if (internalLinkingContent) {
                    // Make sure all tabs are visible
                    var tabs = ["Dashboard", "Settings", "Link Analysis", "AI Suggestions", "Click Analytics", "Link Map"];
                    var tabButtons = document.querySelectorAll("button[role=\"tab\"]");

                    tabButtons.forEach(function(button) {
                        var buttonText = button.textContent;
                        tabs.forEach(function(tabName) {
                            if (buttonText.includes(tabName)) {
                                button.style.display = "flex";
                                button.disabled = false;

                                // Add NEW badge to new features
                                if (tabName === "AI Suggestions" || tabName === "Click Analytics") {
                                    if (!button.querySelector(".prorank-badge-new")) {
                                        var badge = document.createElement("span");
                                        badge.className = "prorank-badge-new";
                                        badge.style.cssText = "background: #ff6b6b; color: white; font-size: 9px; padding: 2px 6px; border-radius: 10px; margin-left: 5px; font-weight: bold;";
                                        badge.textContent = "NEW";
                                        button.appendChild(badge);
                                    }
                                }
                            }
                        });
                    });

                    clearInterval(checkInterval);
                }
            }, 500);

            // Stop checking after 10 seconds
            setTimeout(function() { clearInterval(checkInterval); }, 10000);
        })();
        ';
    }
}

// Initialize immediately
InternalLinkingStyles::init();
