<?php
/**
 * Editor Integration for Internal Linking
 * 
 * Loads the Gutenberg editor panel for internal linking suggestions
 */

declare(strict_types=1);

namespace ProRank\SEO\Admin;

defined( 'ABSPATH' ) || exit;

/**
 * EditorIntegration class
 */
class EditorIntegration {
    
    /**
     * Initialize editor integration
     */
    public static function init(): void {
        add_action('enqueue_block_editor_assets', [__CLASS__, 'enqueue_editor_assets']);
    }
    
    /**
     * Enqueue editor assets
     */
    public static function enqueue_editor_assets(): void {
        // Check if internal linking module is enabled
        $plugin = \ProRank\SEO\Plugin::get_instance();
        if ($plugin) {
            $module_manager = $plugin->modules();
            if (!$module_manager || !$module_manager->is_module_enabled('internal-linking')) {
                return;
            }
        } else {
            // If we can't get the plugin instance, just skip the check
            // and let the feature load if available
        }
        
        // Check user permissions
        if (!current_user_can('edit_posts')) {
            return;
        }
        
        $asset_file = PRORANK_PLUGIN_DIR . 'build/editor-internal-linking.asset.php';
        
        if (file_exists($asset_file)) {
            $asset = require $asset_file;
            
            wp_enqueue_script(
                'prorank-editor-internal-linking',
                PRORANK_PLUGIN_URL . 'build/editor-internal-linking.js',
                $asset['dependencies'],
                $asset['version'],
                true
            );
            
            wp_localize_script('prorank-editor-internal-linking', 'prorankEditor', [
                'apiUrl' => home_url('/wp-json/prorank-seo/v1'),
                'nonce' => wp_create_nonce('wp_rest'),
                'postId' => get_the_ID(),
                'pluginUrl' => PRORANK_PLUGIN_URL,
                'i18n' => [
                    'panelTitle' => __('Internal Linking', 'prorank-seo'),
                    'loading' => __('Loading suggestions...', 'prorank-seo'),
                    'noSuggestions' => __('No suggestions available', 'prorank-seo'),
                    'addLink' => __('Add Link', 'prorank-seo'),
                    'relevance' => __('Relevance', 'prorank-seo'),
                    'refresh' => __('Refresh', 'prorank-seo'),
                ],
            ]);
            
            // Enqueue styles if they exist
            $style_file = PRORANK_PLUGIN_DIR . 'build/editor-internal-linking.css';
            if (file_exists($style_file)) {
                wp_enqueue_style(
                    'prorank-editor-internal-linking',
                    PRORANK_PLUGIN_URL . 'build/editor-internal-linking.css',
                    [],
                    $asset['version']
                );
            }
        }
    }
}