<?php
// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.SlowDBQuery, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in, WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
/**
 * AJAX Handlers for ProRank SEO
 *
 * Handles all AJAX requests from the admin interface
 * Ensures all toggles and settings actually work
 *
 * @package ProRank\Admin
 * @since 3.0.0
 */

declare(strict_types=1);

namespace ProRank\Admin;

defined( 'ABSPATH' ) || exit;

use ProRank\Core\Cache\PageCacheManager;
use ProRank\SEO\Core\Services\LlmsTxtService;
use ProRank\Modules\Performance\ImageOptimizationModule;

class AjaxHandlers {

    /**
     * Initialize AJAX handlers
     */
    public function __construct() {
        // Universal settings handler
        add_action('wp_ajax_prorank_save_setting', [$this, 'handle_save_setting']);

        // Module management
        add_action('wp_ajax_prorank_toggle_module', [$this, 'handle_toggle_module']);

        // LLMs optimization - Individual handlers for each toggle
        add_action('wp_ajax_prorank_save_llms_setting', [$this, 'handle_save_llms_setting']);
        add_action('wp_ajax_prorank_schedule_llms_generation', [$this, 'handle_schedule_llms_generation']);
        add_action('wp_ajax_prorank_clear_llms_schedule', [$this, 'handle_clear_llms_schedule']);
        add_action('wp_ajax_prorank_llms_generate', [$this, 'handle_llms_generate']);
        add_action('wp_ajax_prorank_llms_status', [$this, 'handle_llms_status']);

        // Cache management
        add_action('wp_ajax_prorank_cache_control', [$this, 'handle_cache_control']);
        add_action('wp_ajax_prorank_clear_cache', [$this, 'handle_clear_cache']);

        // Database optimization
        add_action('wp_ajax_prorank_optimize_database', [$this, 'handle_optimize_database']);

        // Image optimization
        add_action('wp_ajax_prorank_optimize_images', [$this, 'handle_optimize_images']);
    }

    /**
     * Universal setting save handler
     * Makes sure ALL toggles and settings actually save
     */
    public function handle_save_setting() {
        // Security check
        if (!check_ajax_referer('prorank_nonce', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $setting = isset($_POST['setting']) ? sanitize_text_field( wp_unslash( $_POST['setting'] ) ) : '';
        $value = '';
        if ( isset( $_POST['value'] ) ) {
            if ( is_array( $_POST['value'] ) ) {
                $value = array_map( 'sanitize_text_field', wp_unslash( $_POST['value'] ) );
            } else {
                $value = sanitize_text_field( wp_unslash( $_POST['value'] ) );
            }
        }

        if (empty($setting)) {
            wp_send_json_error(['message' => __('Setting name required', 'prorank-seo')]);
            return;
        }

        // Handle boolean values
        if ($value === 'true' || $value === 'false') {
            $value = ($value === 'true');
        }

        // Save to database
        $option_name = 'prorank_' . $setting;
        update_option($option_name, $value);

        // Execute feature-specific actions
        $this->execute_feature_action($setting, $value);

        wp_send_json_success([
            'message' => __('Setting saved successfully', 'prorank-seo'),
            'setting' => $setting,
            'value' => $value
        ]);
    }

    /**
     * Handle module toggle
     */
    public function handle_toggle_module() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $module_slug = isset( $_POST['module'] ) ? sanitize_text_field( wp_unslash( $_POST['module'] ) ) : '';
        $enabled = isset( $_POST['enabled'] ) ? filter_var( wp_unslash( $_POST['enabled'] ), FILTER_VALIDATE_BOOLEAN ) : false;

        if (empty($module_slug)) {
            wp_send_json_error(['message' => __('Module slug required', 'prorank-seo')]);
            return;
        }

        // Get current modules
        $modules = get_option('prorank_modules', []);

        // Update module status
        $modules[$module_slug] = [
            'enabled' => $enabled,
            'updated_at' => current_time('mysql')
        ];

        update_option('prorank_modules', $modules);

        // Activate/deactivate module features
        if ($enabled) {
            $this->activate_module_features($module_slug);
        } else {
            $this->deactivate_module_features($module_slug);
        }

        wp_send_json_success([
            'message' => $enabled
                ? __('Module enabled successfully', 'prorank-seo')
                : __('Module disabled successfully', 'prorank-seo'),
            'module' => $module_slug,
            'enabled' => $enabled
        ]);
    }

    /**
     * Save individual LLMs setting
     */
    public function handle_save_llms_setting() {
        // Security check
        if (!check_ajax_referer('prorank_ajax', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $setting = isset( $_POST['setting'] ) ? sanitize_text_field( wp_unslash( $_POST['setting'] ) ) : '';
        $raw_value = isset( $_POST['value'] ) ? sanitize_text_field( wp_unslash( $_POST['value'] ) ) : '';
        $value = $raw_value === 'true' || $raw_value === true;

        // Get current settings using standardized key
        $llms_settings = get_option('prorank_llms_txt_settings', [
            'enabled' => false,
            'mode' => 'automatic',
            'pages' => [],
            'include_posts' => true,
            'include_pages' => true,
            'include_products' => true,
            'max_pages' => 20
        ]);

        // Update specific setting
        $llms_settings[$setting] = $value;
        update_option('prorank_llms_txt_settings', $llms_settings);

        // Log for debugging (dev only)
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            prorank_log('[ProRank LLMs] Setting ' . $setting . ' changed to ' . ($value ? 'true' : 'false'));
        }

        wp_send_json_success([
            'message' => __('Setting saved', 'prorank-seo'),
            'setting' => $setting,
            'value' => $value
        ]);
    }

    /**
     * Schedule weekly LLMs generation
     */
    public function handle_schedule_llms_generation() {
        // Security check
        if (!check_ajax_referer('prorank_ajax', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        // Clear any existing schedule
        wp_clear_scheduled_hook('prorank_llmstxt_regenerate');

        // Schedule weekly generation
        if (!wp_next_scheduled('prorank_llmstxt_regenerate')) {
            wp_schedule_event(time() + WEEK_IN_SECONDS, 'weekly', 'prorank_llmstxt_regenerate');
        }

        // Also generate immediately
        $this->generate_llms_files();

        wp_send_json_success([
            'message' => __('LLMs generation scheduled and files generated', 'prorank-seo'),
            'next_run' => wp_next_scheduled('prorank_llmstxt_regenerate')
        ]);
    }

    /**
     * Clear LLMs generation schedule
     */
    public function handle_clear_llms_schedule() {
        // Security check
        if (!check_ajax_referer('prorank_ajax', 'nonce', false)) {
            wp_send_json_error(['message' => __('Security check failed', 'prorank-seo')]);
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        // Clear scheduled hook
        wp_clear_scheduled_hook('prorank_llmstxt_regenerate');

        // Also delete the generated files
        $llms_dir = $this->get_llms_dir();

        if (file_exists($llms_dir . '/llms.txt')) {
            wp_delete_file($llms_dir . '/llms.txt');
        }
        if (file_exists($llms_dir . '/llms-full.txt')) {
            wp_delete_file($llms_dir . '/llms-full.txt');
        }

        delete_transient('prorank_llms_txt');
        delete_transient('prorank_llms_full_txt');

        wp_send_json_success([
            'message' => __('LLMs generation schedule cleared and files removed', 'prorank-seo')
        ]);
    }

    /**
     * Generate LLMs files immediately
     */
    private function generate_llms_files() {
        $settings = get_option('prorank_llms_txt_settings', []);
        $enabled = is_array($settings) && !empty($settings['enabled']);
        if (!$enabled) {
            return false;
        }

        $service = new LlmsTxtService();
        $index_content = $service->generate_index();
        $full_content = $service->generate_full();

        if (!$index_content || !$full_content) {
            return false;
        }

        $this->store_llms_files($index_content, $full_content);
        update_option('prorank_llms_last_generated', time());

        return true;
    }

    /**
     * Store llms.txt contents locally for virtual serving.
     *
     * @param string $index_content
     * @param string $full_content
     */
    private function store_llms_files(string $index_content, string $full_content): void {
        $llms_dir = $this->get_llms_dir();

        if (!file_exists($llms_dir)) {
            wp_mkdir_p($llms_dir);
        }

        global $wp_filesystem;
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ($wp_filesystem) {
            $wp_filesystem->put_contents($llms_dir . '/llms.txt', $index_content, FS_CHMOD_FILE);
            $wp_filesystem->put_contents($llms_dir . '/llms-full.txt', $full_content, FS_CHMOD_FILE);
        }

        set_transient('prorank_llms_txt', $index_content, WEEK_IN_SECONDS);
        set_transient('prorank_llms_full_txt', $full_content, WEEK_IN_SECONDS);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            prorank_log('[ProRank LLMs] Files generated successfully at ' . $llms_dir);
        }
    }

    /**
     * Get LLMs storage directory.
     *
     * @return string
     */
    private function get_llms_dir(): string {
        $upload_dir = wp_upload_dir();
        $legacy_dir = trailingslashit($upload_dir['basedir']) . 'prorank-llms';

        if (is_dir($legacy_dir)) {
            return $legacy_dir;
        }

        return trailingslashit($upload_dir['basedir']) . 'prorank-seo/llms';
    }

    /**
     * Handle LLMs.txt generation
     */
    public function handle_llms_generate() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        try {
            // Generate llms.txt files using new service
            $service = new LlmsTxtService();
            $index_content = $service->generate_index();
            $full_content = $service->generate_full();

            if ($index_content && $full_content) {
                $this->store_llms_files($index_content, $full_content);

                // Update last generated timestamp
                update_option('prorank_llms_last_generated', current_time('timestamp'));

                wp_send_json_success([
                    'message' => __('LLMs files generated successfully', 'prorank-seo'),
                    'urls' => [
                        'index' => home_url('/llms.txt'),
                        'full' => home_url('/llms-full.txt')
                    ]
                ]);
            } else {
                wp_send_json_error([
                    'message' => __('Failed to generate LLMs files', 'prorank-seo')
                ]);
            }
        } catch (\Exception $e) {
            wp_send_json_error([
                'message' => esc_html($e->getMessage())
            ]);
        }
    }

    /**
     * Get LLMs.txt status
     */
    public function handle_llms_status() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $last_generated = get_option('prorank_llms_last_generated', 0);
        $next_update = $last_generated + (7 * DAY_IN_SECONDS); // Weekly updates

        wp_send_json_success([
            'last_generated' => $last_generated,
            'last_generated_formatted' => $last_generated ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_generated) : null,
            'next_update_formatted' => $last_generated ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $next_update) : null,
            'urls' => [
                'index' => home_url('/llms.txt'),
                'full' => home_url('/llms-full.txt')
            ],
            'cache_exists' => [
                'index' => file_exists($this->get_llms_dir() . '/llms.txt'),
                'full' => file_exists($this->get_llms_dir() . '/llms-full.txt')
            ]
        ]);
    }

    /**
     * Handle cache control
     */
    public function handle_cache_control() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $action = isset( $_POST['cache_action'] ) ? sanitize_text_field( wp_unslash( $_POST['cache_action'] ) ) : '';

        switch ($action) {
            case 'enable':
                $this->enable_page_cache();
                wp_send_json_success(['message' => __('Page cache enabled', 'prorank-seo')]);
                break;

            case 'disable':
                $this->disable_page_cache();
                wp_send_json_success(['message' => __('Page cache disabled', 'prorank-seo')]);
                break;

            case 'clear':
                $this->clear_page_cache();
                wp_send_json_success(['message' => __('Cache cleared successfully', 'prorank-seo')]);
                break;

            default:
                wp_send_json_error(['message' => __('Invalid cache action', 'prorank-seo')]);
        }
    }

    /**
     * Handle cache clearing
     */
    public function handle_clear_cache() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $type = isset( $_POST['cache_type'] ) ? sanitize_text_field( wp_unslash( $_POST['cache_type'] ) ) : 'all';

        try {
            $cleared = [];

            if ($type === 'all' || $type === 'page') {
                $this->clear_page_cache();
                $cleared[] = 'page';
            }

            if ($type === 'all' || $type === 'object') {
                wp_cache_flush();
                $cleared[] = 'object';
            }

            if ($type === 'all' || $type === 'transient') {
                $this->clear_transients();
                $cleared[] = 'transient';
            }

            wp_send_json_success([
                /* translators: %s: list of cache types cleared */
                'message' => sprintf(
                    /* translators: %s: placeholder value */
                    __('Cleared %s cache', 'prorank-seo'),
                    implode(', ', $cleared)
                ),
                'cleared' => $cleared
            ]);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => esc_html($e->getMessage())]);
        }
    }

    /**
     * Handle database optimization
     */
    public function handle_optimize_database() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        global $wpdb;

        try {
            $optimized_tables = [];

            // Get all database tables
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);

            foreach ($tables as $table) {
                $table_name = $table[0];

                // Only optimize WordPress tables
                if (strpos($table_name, $wpdb->prefix) === 0) {
                    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
                    $wpdb->query("OPTIMIZE TABLE `$table_name`");
                    $optimized_tables[] = $table_name;
                }
            }

            // Clean up options
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_%'");

            // Clean up post meta
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
            $wpdb->query("DELETE pm FROM {$wpdb->postmeta} pm LEFT JOIN {$wpdb->posts} p ON p.ID = pm.post_id WHERE p.ID IS NULL");

            wp_send_json_success([
                /* translators: %d: number of database tables optimized */
                'message' => sprintf(
                    /* translators: %s: placeholder value */
                    __('Optimized %d tables', 'prorank-seo'),
                    count($optimized_tables)
                ),
                'tables' => $optimized_tables
            ]);
        } catch (\Exception $e) {
            wp_send_json_error(['message' => esc_html($e->getMessage())]);
        }
    }

    /**
     * Handle image optimization
     */
    public function handle_optimize_images() {
        check_ajax_referer('prorank_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Insufficient permissions', 'prorank-seo')]);
            return;
        }

        $image_id = intval($_POST['image_id'] ?? 0);

        if (!$image_id) {
            wp_send_json_error(['message' => __('Image ID required', 'prorank-seo')]);
            return;
        }

        try {
            $optimizer = new \ProRank\Core\ImageOptimizer();
            $result = $optimizer->optimize($image_id);

            if ($result['success']) {
                wp_send_json_success([
                    'message' => __('Image optimized successfully', 'prorank-seo'),
                    'saved' => $result['saved_bytes'],
                    'percentage' => $result['percentage']
                ]);
            } else {
                wp_send_json_error(['message' => $result['message']]);
            }
        } catch (\Exception $e) {
            wp_send_json_error(['message' => esc_html($e->getMessage())]);
        }
    }

    /**
     * Execute feature-specific actions when settings change
     */
    private function execute_feature_action($setting, $value) {
        switch ($setting) {
            case 'page_cache':
            case 'cache_enabled':
                if ($value) {
                    $this->enable_page_cache();
                } else {
                    $this->disable_page_cache();
                }
                break;

            case 'browser_cache_enabled':
                if ($value) {
                    $this->enable_browser_cache();
                } else {
                    $this->disable_browser_cache();
                }
                break;

            case 'gzip_compression':
                if ($value) {
                    $this->enable_gzip_compression();
                } else {
                    $this->disable_gzip_compression();
                }
                break;

            case 'lazy_load_images':
                if ($value) {
                    add_filter('wp_lazy_loading_enabled', '__return_true');
                } else {
                    add_filter('wp_lazy_loading_enabled', '__return_false');
                }
                break;
        }
    }

    /**
     * Enable page caching
     */
    private function enable_page_cache() {
        // Create cache directory if it doesn't exist
        $cache_dir = prorank_page_cache_dir();
        if (!file_exists($cache_dir)) {
            wp_mkdir_p($cache_dir);
        }

        // Update .htaccess for caching rules
        $this->update_htaccess_cache_rules(true);

        update_option('prorank_page_cache_enabled', true);
    }

    /**
     * Disable page caching
     */
    private function disable_page_cache() {
        // Clear existing cache
        $this->clear_page_cache();

        // Remove .htaccess rules
        $this->update_htaccess_cache_rules(false);

        update_option('prorank_page_cache_enabled', false);
    }

    /**
     * Clear page cache
     */
    private function clear_page_cache() {
        $cache_dir = prorank_page_cache_dir();

        if (file_exists($cache_dir)) {
            $this->delete_directory($cache_dir);
            wp_mkdir_p($cache_dir);
        }
    }

    /**
     * Clear transients
     */
    private function clear_transients() {
        global $wpdb;

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_transient_%'");
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Custom table name is safe
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '%_site_transient_%'");
    }

    /**
     * Enable browser caching via .htaccess
     */
    private function enable_browser_cache() {
        $rules = $this->get_browser_cache_rules();
        $this->insert_htaccess_rules($rules, 'ProRank Browser Cache');
    }

    /**
     * Disable browser caching
     */
    private function disable_browser_cache() {
        $this->remove_htaccess_rules('ProRank Browser Cache');
    }

    /**
     * Enable GZIP compression
     */
    private function enable_gzip_compression() {
        $rules = $this->get_gzip_rules();
        $this->insert_htaccess_rules($rules, 'ProRank GZIP');
    }

    /**
     * Disable GZIP compression
     */
    private function disable_gzip_compression() {
        $this->remove_htaccess_rules('ProRank GZIP');
    }

    /**
     * Get browser cache rules for .htaccess
     */
    private function get_browser_cache_rules() {
        return '
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpg "access plus 1 year"
    ExpiresByType image/jpeg "access plus 1 year"
    ExpiresByType image/gif "access plus 1 year"
    ExpiresByType image/png "access plus 1 year"
    ExpiresByType text/css "access plus 1 month"
    ExpiresByType text/javascript "access plus 1 month"
    ExpiresByType application/javascript "access plus 1 month"
    ExpiresByType application/x-shockwave-flash "access plus 1 month"
    ExpiresByType image/x-icon "access plus 1 year"
    ExpiresDefault "access plus 2 days"
</IfModule>';
    }

    /**
     * Get GZIP compression rules
     */
    private function get_gzip_rules() {
        return '
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css
    AddOutputFilterByType DEFLATE text/javascript application/javascript application/x-javascript
    AddOutputFilterByType DEFLATE application/xml application/xhtml+xml application/rss+xml
    AddOutputFilterByType DEFLATE application/font-woff application/font-woff2 application/vnd.ms-fontobject
    AddOutputFilterByType DEFLATE application/x-font-ttf font/opentype image/svg+xml image/x-icon
</IfModule>';
    }

    /**
     * Update .htaccess with cache rules
     */
    private function update_htaccess_cache_rules($enable) {
        if ($enable) {
            $rules = $this->get_browser_cache_rules();
            $this->insert_htaccess_rules($rules, 'ProRank Cache');
        } else {
            $this->remove_htaccess_rules('ProRank Cache');
        }
    }

    /**
     * Insert rules into .htaccess
     */
    private function insert_htaccess_rules($rules, $marker) {
        if (!function_exists('insert_with_markers')) {
            require_once(ABSPATH . 'wp-admin/includes/misc.php');
        }

        $htaccess = prorank_get_home_path() . '.htaccess';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ( $wp_filesystem && $wp_filesystem->is_writable( $htaccess ) ) {
            $lines = explode("\n", trim($rules));
            insert_with_markers($htaccess, $marker, $lines);
        }
    }

    /**
     * Remove rules from .htaccess
     */
    private function remove_htaccess_rules($marker) {
        if (!function_exists('insert_with_markers')) {
            require_once(ABSPATH . 'wp-admin/includes/misc.php');
        }

        $htaccess = prorank_get_home_path() . '.htaccess';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ( $wp_filesystem && $wp_filesystem->is_writable( $htaccess ) ) {
            insert_with_markers($htaccess, $marker, []);
        }
    }

    /**
     * Activate module features
     */
    private function activate_module_features($module_slug) {
        switch ($module_slug) {
            case 'page-cache':
                $this->enable_page_cache();
                break;

            case 'image-optimization':
                update_option('prorank_image_optimization_enabled', true);
                break;

            case 'database-optimization':
                wp_schedule_event(time(), 'weekly', 'prorank_optimize_database');
                break;

            case 'llms-optimization':
                update_option('prorank_llms_enabled', true);
                wp_schedule_event(time(), 'weekly', 'prorank_llmstxt_regenerate');
                break;
        }
    }

    /**
     * Deactivate module features
     */
    private function deactivate_module_features($module_slug) {
        switch ($module_slug) {
            case 'page-cache':
                $this->disable_page_cache();
                break;

            case 'image-optimization':
                update_option('prorank_image_optimization_enabled', false);
                break;

            case 'database-optimization':
                wp_clear_scheduled_hook('prorank_optimize_database');
                break;

            case 'llms-optimization':
                update_option('prorank_llms_enabled', false);
                wp_clear_scheduled_hook('prorank_llmstxt_regenerate');
                break;
        }
    }

    /**
     * Recursively delete a directory
     */
    private function delete_directory($dir) {
        if (!file_exists($dir)) {
            return;
        }

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        $files = array_diff(scandir($dir), ['.', '..']);

        foreach ($files as $file) {
            $path = $dir . '/' . $file;

            if (is_dir($path)) {
                $this->delete_directory($path);
            } else {
                wp_delete_file($path);
            }
        }

        if ( $wp_filesystem ) {
            $wp_filesystem->rmdir( $dir );
        }
    }
}

// Initialize the AJAX handlers
new AjaxHandlers();
